<?php
/**
 * FredT_Request_Info_UserAgentTest test case.
 *
 * @author      Fred
 * @category    FredT Library
 * @package     FredT
 * @version     31/01/2009
 *
 * @uses PHPUnit, need add necessary require for the library
 */

require_once '../UserAgent.php';

class FredT_Request_Info_UserAgentTest extends PHPUnit_Framework_TestCase
{
    /**
     * @var FredT_Request_Info_UserAgent
     */
    private $_infoUA;
    /**
     * @var simplexml_load_file
     */
    private $_oSXE;
    private $_defautMode;
    private $_defautFileCache;

    protected function setUp ()
    {
        parent::setUp();
        $this->_defautMode=FredT_Request_Info_UserAgent::getDetectMode();
        $this->_defautFileCache=FredT_Request_Info_UserAgent::getFileCache();
        FredT_Request_Info_UserAgent::setDetectMode(FredT_Request_Info_UserAgent::MODE_FULL_DETECT);
        $file=dirname(__FILE__) . '/_data/cache.php';
        if (is_file($file)) {
            unlink($file);
        }
        FredT_Request_Info_UserAgent::setFileCache($file);
        $this->_infoUA = new FredT_Request_Info_UserAgent();
        $this->_oSXE = @simplexml_load_file(dirname(__FILE__) . '/_data/UA.xml');
        FredT_Request_Info_UserAgent::unloadReference();
    }

    public function testValeurDefautInitialisation ()
    {
        $this->assertNull($this->_defautFileCache);
        $this->assertEquals($this->_defautMode, FredT_Request_Info_UserAgent::MODE_FAST_DETECT, 'mode par defaut est pas FAST');
    }

    public function testUsageDuFichierCache ()
    {
        $file=dirname(__FILE__) . '/_data/testReadCache.php';
        $this->assertFileExists($file);
        FredT_Request_Info_UserAgent::setFileCache($file);
        $cache=FredT_Request_Info_UserAgent::getReference();
        $this->assertArrayHasKey('test du cache', $cache['agent']);
    }

    public function testFichierCacheCorrectementCree ()
    {
        $file=dirname(__FILE__) . '/_data/testCreeCache.php';
        if (is_file($file)) {
            unlink($file);
        }

        FredT_Request_Info_UserAgent::setFileCache($file);
        $this->assertFileExists($file);
        $cache=include($file);
        $ref=include('FredT/Request/Info/Data/refUA.php');
        $this->assertEquals(count($ref), count($cache));
        $this->assertEquals(count($ref['agent']), count($cache['agent']));
        $this->assertEquals(count($ref['system']), count($cache['system']));
        unlink($file);
    }

    public function testGettersAvecGetInfos ()
    {
        $aCompare = array (
          'user agent' => 'Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; {E42670B0-BB3D-4A66-A450-33B64CE9BB8E}; FREE)',
          'agent type' => 'browser',
          'agent sub-type' => '',
          'agent name' => 'Internet Explorer',
          'agent version' => '6.0',
          'agent description' => null,
          'system name' => 'Windows',
          'system version' => 'NT',
          'system type' => 'os',
          'system sub-type' => '',
          'system description' => null,
          'NetClr' => false,
        );

        $aTest=$this->_infoUA->setUserAgent('Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; {E42670B0-BB3D-4A66-A450-33B64CE9BB8E}; FREE)')
                      ->getInfos();
        foreach ($aCompare as $key => $value) {
        	$this->assertEquals($aTest[$key], $value,$key . ' value not equals');
        }
        //var_export($aTest);
//        $this->assertSame($aCompare, $aTest);
    }

    public function testGetOS ()
    {
        $this->_infoUA->setUserAgent('Mozilla/5.0 (X11; U; Linux i686; fr; rv:1.9.0.3) Gecko/2008101315 Ubuntu/8.10 (intrepid) Firefox/3.0.3');
        $this->assertEquals('Linux', $this->_infoUA->getSystem());
        $this->assertEquals('Ubuntu', $this->_infoUA->getSystemVersion());
    }

    public function testSystemIs ()
    {
        $aTest =array(
            0=>'Mozilla/5.0 (compatible; Googlebot/2.1;+http://www.google.com/bot.html)',
            1=>'Mozilla/4.0 (compatible)',
            //2=>'',//dont have in my stats
            3=>'Nokia6680/1.0 ((4.04.07) SymbianOS/8.0 Series60/2.6 Profile/MIDP-2.0 Configuration/CLDC-1.1 (botmobi find.mobi/bot.html find@mtld.mobi) )',
            4=>'Mozilla/4.0 SonyEricssonW910iv/R1CA Browser/NetFront/3.4 Profile/MIDP-2.1 Configuration/CLDC-1.1',
            5=>'Nokia6682/2.0 (3.01.1) SymbianOS/8.0 Series60/2.6 Profile/MIDP-2.0 configuration/CLDC-1.1 UP.Link/6.3.0.0.0 (compatible;YahooSeeker/M1A1-R2D2; ',
            6=>'Mozilla/4.0 SonyEricssonW910iv/R1CA Browser/NetFront/3.4 Profile/MIDP-2.1 Configuration/CLDC-1.1',
            7=>'Mozilla/4.0 WebTV/2.8 (compatible; MSIE 4.0)',
        );
        $result=$this->_infoUA->setUserAgent($aTest[0])->isUnknowSystem();
        $this->assertTrue($result, $aTest[0] . ' is Unknow System');
        $result=$this->_infoUA->setUserAgent($aTest[1])->isUnknowSystem();
        $this->assertTrue($result, $aTest[1] . ' is Unknow System');
        //$result=$this->_infoUA->setUserAgent($aTest[2])->isConsoleSystem();
        //$this->assertTrue($result, $aTest[2] . ' is Console System');
        $result=$this->_infoUA->setUserAgent($aTest[3])->isMobilSystem();
        $this->assertTrue($result, $aTest[3] . ' is Mobil System');
        $result=$this->_infoUA->setUserAgent($aTest[4])->isMobilSystem();
        $this->assertTrue($result, $aTest[4] . ' is Mobil System');
        $result=$this->_infoUA->setUserAgent($aTest[5])->isMobilSystem();
        $this->assertTrue($result, $aTest[5] . ' is Mobil System');
        $result=$this->_infoUA->setUserAgent($aTest[6])->isMobilSystem();
        $this->assertTrue($result, $aTest[6] . ' is Mobil System');
        $result=$this->_infoUA->setUserAgent($aTest[7])->isTvSystem();
        $this->assertTrue($result, $aTest[7] . ' is TV System');
    }

    public function testModeFullRapideLightEtPerformances ()
    {
        $aModes=array(
                   FredT_Request_Info_UserAgent::MODE_FULL_DETECT,
        FredT_Request_Info_UserAgent::MODE_FAST_DETECT,
                   FredT_Request_Info_UserAgent::MODE_LIGHT_DETECT,
               );
        $aResult=array();
        foreach ($aModes as $mode) {
            $aResult[$mode]=array();
            $time_start = microtime(true);
            FredT_Request_Info_UserAgent::setDetectMode($mode);
            $ref=FredT_Request_Info_UserAgent::getReference();
            $aResult[$mode]['agents'] = count($ref['agent']);
            $aResult[$mode]['timeInit'] = microtime(true) - $time_start;
            $time_start = microtime(true);
            foreach ($this->_oSXE as $val) {
                $this->_infoUA->setUserAgent((string) $val->navigateur);
                $this->_infoUA->getInfos();
            }
            $aResult[$mode]['timeDetect'] = microtime(true) - $time_start;
        }
// Performances Result
//echo ' Performances Result:'."\n"; var_dump($aResult);
        $this->assertLessThan($aResult[3]['agents'],$aResult[2]['agents']);
        $this->assertLessThan($aResult[2]['agents'],$aResult[1]['agents']);
        $this->assertLessThan($aResult[3]['timeDetect'],$aResult[2]['timeDetect']);
        $this->assertLessThan($aResult[2]['timeDetect'],$aResult[1]['timeDetect']);
        $this->assertLessThan($aResult[3]['timeInit'],$aResult[1]['timeInit']);
    }

    public function testisUnknow ()
    {
        $aTest=array(
            'panscient.com',
            'NA',
            'sohu-search',
            'PHP/4.2.3',
            '-',
        );
        foreach ($aTest as $val) {
        	$this->_infoUA->setUserAgent($val);
        	$this->assertTrue($this->_infoUA->isUnknow());
        }
    }

    public function testIsUndesirable ()
    {
        $aTest=array(
            'WebZIP/5.0 (http://www.spidersoft.com)',
            'TurnitinBot/2.1 (http://www.turnitin.com/robot/crawlerinfo.html)',
            'Mozilla/4.5 (compatible; HTTrack 3.0x; Windows 98)',
            'TurnitinBot/2.0 http://www.turnitin.com/robot/crawlerinfo.html',
            'Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1; Wanadoo 6.1; Wysigot 5.51)',
            'Mozilla/3.0 (compatible; WebCapture 2.0; Auto; Windows)',
            'Wget/1.8.2',
            'Wget/1.10.2 (Red Hat modified)',
            'Mozilla/3.0 (compatible; Indy Library)',
            'Wget/1.10.1 (Red Hat modified)',
            'wget',
        );
        foreach ($aTest as $val) {
            $this->_infoUA->setUserAgent($val);
            if (! $this->_infoUA->isUndesirable()) echo $val.' - '. $this->_infoUA->getType()."\n";
            $this->assertTrue($this->_infoUA->isUndesirable());
        }
    }

    public function testIsGoogleBot ()
    {
        $aTest=array(
            'Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)',
            'Googlebot/2.1 (+http://www.googlebot.com/bot.html)',
            'Googlebot/2.1 (+http://www.google.com/bot.html)',
            'Googlebot/Test',
            'Mozilla/4.0 (MobilePhone SCP-5500/US/1.0) NetFront/3.0 MMP/2.0 FAKE (compatible; Googlebot/2.1; +http://www.google.com/bot.html)',
            'Mozilla/5.0 (compatible; Googlebot/2.1;+http://www.google.com/bot.html)',
            ' Googlebot/Test (+http://www.googlebot.com/bot.html)',
        );
        foreach ($aTest as $val) {
        	$this->_infoUA->setUserAgent($val);
        	$this->assertTrue($this->_infoUA->isGoogleBot());
        }
    }

    public function testIsBrowserEtIsBotAvecListBotDetectHasBrowser ()
    {
        $aEssai=array (
          	'Mozilla/5.0 (compatible; Yahoo! Slurp; http://help.yahoo.com/help/us/ysearch/slurp)',
            'Mozilla/4.0 (compatible; MSIE 5.0; Windows 95) VoilaBot BETA 1.2 (http://www.voila.com/)',
            'Mozilla/5.0 (Windows; U; Windows NT 5.1; fr; rv:1.8.1) VoilaBot BETA 1.2 (support.voilabot@orange-ftgroup.com)',
            'Mozilla/4.0 (compatible; MSIE 6.0 compatible; Asterias Crawler v4; +http://www.singingfish.com/help/spider.html; webmaster@singingfish.com); SpiderThr',
            'Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1; iOpus-I-M; .NET CLR 1.1.4322; .NET CLR 2.0.50727; InfoPath.1; FDM; Badongo 2.0.0)',
        );
        foreach ($aEssai as $val) {
            $this->_infoUA->setUserAgent($val);
            $this->assertFalse($this->_infoUA->isBrowser());
            $this->assertTrue($this->_infoUA->isBot());
        }
    }

    public function testIsFirefox ()
    {
        $aTest=array(
            'Mozilla/5.0 (Windows; U; Windows NT 5.2; en-US; rv:1.9.0.1) Gecko/2008070208 Firefox/3.0.1',
            'Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.8.0.4) Gecko/20060508 Firefox/1.5.0.4',
        	'Mozilla/5.0 (Windows; U; Windows NT 5.1; en-GB; rv:1.8.1) Gecko/20061010 Firefox/2.0',
        );
        foreach ($aTest as $val) {
        	$this->assertTrue($this->_infoUA->setUserAgent($val)->isFirefox());
        }
    }
    public function testIsIE ()
    {
        $aTest=array(
            'Microsoft Internet Explorer IE/6.0 Windows',
        	'Mozilla/4.0 (compatible; MSIE 999.1; Unknown)',
            'Mozilla/4.0 (compatible; MSIE 5.01; Windows 98; 981)',
            'Mozilla/2.0 (compatible; MSIE 3.02; Windows CE; 240x320)',
            'Mozilla/4.0 (compatible; MSIE 5.0b1; Mac_PowerPC)',

            'Mozilla/4.0 WebTV/2.8 (compatible; MSIE 4.0)',
        );
        foreach ($aTest as $val) {
        	$this->assertTrue($this->_infoUA->setUserAgent($val)->isIE());
        }
    }

    public function testIsOpera ()
    {
        $aTest=array(
            'Opera/9.24 (Windows NT 5.1; U; en)',
            'Opera - 7.23 -  === Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1) Opera 7.23  [fr]',
            'Mozilla/4.0 (compatible; MSIE 6.0; MSIE 5.5; Windows NT 5.0) Opera 7.02 Bork-edition [en]',
        );
        foreach ($aTest as $val) {
        	$this->assertTrue($this->_infoUA->setUserAgent($val)->isOpera());
        }

    }
    public function testIsNET_CLR ()
    {
        $aTest=array(
            'Mozilla/4.0 (compatible; MSIE 5.0; Windows NT 4.1; SV1; .NET CLR 2.0.10707)',
            'Internet Explorer - 7.0 -  === Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1; .NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.04506.30; .NET CLR 3.0.04506.648; .NET CLR 3.5.',
            'Internet Explorer - 6.0 -  === Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1; FunWebProducts; Wanadoo 7.1 ; NaviWoo1.1; .NET CLR 1.1.4322; SpamBlockerUtility 4.8.4)',
            'Internet Explorer - 6.0 -  === Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; Wanadoo 5.5; .NET CLR 1.0.3705; .NET CLR 1.1.4322)',
            'Mozilla/4.0 (compatible; MSIE 6.0; AOL 7.0; Windows NT 5.1; .NET CLR 1.1.4322)',
            'Mozilla/5.0 (Windows; U; Windows NT 5.2; fr; rv:1.9.0.5) Gecko/2008120122 Firefox/3.0.5 (.NET CLR 3.5.30729)',

        );
        foreach ($aTest as $val) {
        	$this->assertTrue($this->_infoUA->setUserAgent($val)->isNET_CLR());
        }
    }
    public function testAccessorsUserAgent ()
    {
        $ua='msnbot/1.0 (+http://search.msn.com/msnbot.htm)';
        $ret=$this->_infoUA->setUserAgent($ua);
        $this->assertType(get_class($this->_infoUA),$ret);
        $this->assertSame($ua,$this->_infoUA->getUserAgent());
    }
}

