<?php
/* * *******************************************************************************
 * The content of this file is subject to the ITS4YouSignature license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 * ****************************************************************************** */

class ITS4YouSignature_Record_Model extends Vtiger_Record_Model
{
    const HTML_VARIABLE = 'ITS4YouSignatureHTML';
    const IMAGE_VARIABLE = 'ITS4YouSignatureImage';
    const LABEL_VARIABLE = '$PDF_SIGNATURE$';

    /**
     * @param string $image
     * @return int
     * @throws AppException
     * @throws Exception
     */
    public function generatePDF($image)
    {
        $this->validateFields(['language', 'source_module', 'source_record', 'template']);

        if (!class_exists('mPDF')) {
            require_once('modules/PDFMaker/resources/mpdf/mpdf.php');
        }

        vglobal(self::IMAGE_VARIABLE, $image);
        $PDFMaker = new PDFMaker_PDFMaker_Model();
        $language = $this->isEmpty('language') ? Vtiger_Language_Handler::getLanguage() : $this->get('language');
        $sourceModule = $this->get('source_module');
        $sourceRecord = intval($this->get('source_record'));
        $template_ids = intval($this->get('template'));
        $contactId = intval($this->get('recipientId'));
        $forView = 'Detail';
        /** @var $moduleModel ITS4YouSignature_Module_Model */
        $moduleModel = $this->getModule();
        $oldRequest = $_REQUEST;
        $_REQUEST = [
            'module' => 'PDFMaker',
            'action' => 'SaveIntoDocuments',
            'pmodule' => $sourceModule,
            'pid' => $sourceRecord,
            'record' => $sourceRecord,
            'forview' => $forView,
            'language' => $language,
            'template_ids' => $template_ids,
            'assigned_user_id' => $this->get('assigned_user_id'),
            'filelocationtype' => 'I',
            'folderid' => $moduleModel->getDocumentFolder()->getId(),
            'notes_title' => 'Signature',
            'parentid' => $sourceRecord,
        ];
        $this->assignPreContent($_REQUEST);
        $request = new Vtiger_Request($_REQUEST, $_REQUEST);
        $documentId = $this->createDocument($request);
        $focus = $this->createNotesRelations($documentId, $sourceRecord, $contactId);
        $PDFMaker->createPDFAndSaveFile($request, $template_ids, $focus, [$sourceRecord], '', $sourceModule, $language);
        $this->setRelation($documentId, 'Documents');
        $this->setRelationForModule($sourceRecord, $sourceModule, $documentId, 'Documents');
        $this->updateNotesTitle($documentId);

        $_REQUEST = $oldRequest;

        return $documentId;
    }

    /**
     * @param Vtiger_Request $request
     * @return int
     */
    public function createDocument(Vtiger_Request $request)
    {
        $moduleName = 'Documents';
        $moduleModel = Vtiger_Module_Model::getInstance($moduleName);
        $recordModel = Vtiger_Record_Model::getCleanInstance($moduleName);
        $recordModel->set('filename', 'Signature.pdf');
        $recordModel->set('filetype', 'application/pdf');
        $recordModel->set('fileversion', 'I');
        $recordModel->set('filestatus', 'on');
        $recordModel->set('parentid', $request->get('parentid'));

        foreach ($moduleModel->getFields() as $fieldName => $fieldModel) {
            if ($request->has($fieldName)) {
                $fieldValue = $request->get($fieldName, null);
            } else {
                $fieldValue = $fieldModel->getDefaultFieldValue();
            }

            $fieldDataType = $fieldModel->getFieldDataType();

            if ('time' === $fieldDataType) {
                $fieldValue = Vtiger_Time_UIType::getTimeValueWithSeconds($fieldValue);
            }
            if (!empty($fieldValue)) {
                if (!is_array($fieldValue)) {
                    $fieldValue = trim($fieldValue);
                }

                $recordModel->set($fieldName, $fieldValue);
            }
        }

        $recordModel->save();

        return intval($recordModel->getId());
    }

    /**
     * @throws Exception
     * @return object
     */
    public function createNotesRelations($documentId, $sourceRecord, $contactId)
    {
        $moduleName = 'Documents';
        $focus = CRMEntity::getInstance($moduleName);
        $focus->retrieve_entity_info($documentId, $moduleName);
        $focus->id = $documentId;
        $focus->parentid = $sourceRecord;
        $focus->insertintonotesrel($sourceRecord, $documentId);

        if ($contactId) {
            $focus->insertintonotesrel($contactId, $documentId);
        }

        return $focus;
    }

    /**
     * @param int $record
     */
    public function updateNotesTitle($record)
    {
        $record = Vtiger_Record_Model::getInstanceById($record, 'Documents');
        $record->set('mode', 'edit');
        $record->set('notes_title', substr($record->get('filename'), 0, -4) . ' - signed');
        $record->save();
    }

    /**
     * @param $fields
     * @throws AppException
     */
    public function validateFields($fields)
    {
        foreach ($fields as $field) {
            if ($this->isEmpty($field)) {
                throw new AppException(vtranslate('LBL_REQUIRED_FIELD', $this->getModuleName()) . $field);
            }
        }
    }

    /**
     * @return string
     */
    public function getPDFPreviewURL()
    {
        return $this->getSignatureUrl() . '&mode=showPreviewPDF';
    }

    /**
     * @param $id
     * @param $module
     */
    public function setRelation($id, $module)
    {
        $sourceModuleModel = $this->getModule();
        $relatedModuleModel = Vtiger_Module_Model::getInstance($module);

        if ($relatedModuleModel) {
            $relationModel = Vtiger_Relation_Model::getInstance($sourceModuleModel, $relatedModuleModel);

            if ($relationModel) {
                $relationModel->addRelation($this->getId(), $id);
            }
        }
    }

    public function setRelationForModule($sourceId, $sourceModule, $relatedId, $relatedModule)
    {
        if (empty($sourceId) || empty($sourceModule)) {
            return;
        }

        $sourceModuleModel = Vtiger_Module_Model::getInstance($sourceModule);
        $relatedModuleModel = Vtiger_Module_Model::getInstance($relatedModule);

        if ($relatedModuleModel) {
            $relationModel = Vtiger_Relation_Model::getInstance($sourceModuleModel, $relatedModuleModel);

            if ($relationModel) {
                $relationModel->addRelation($sourceId, $relatedId);
            }
        }
    }

    public function getRelationQuery($relatedModule, $parentModule = 'ITS4YouSignature')
    {
        vglobal('currentModule', $parentModule);

        $parentModuleModel = Vtiger_Module_Model::getInstance($parentModule);
        $relatedModuleModel = Vtiger_Module_Model::getInstance($relatedModule);
        $relatedList = Vtiger_Relation_Model::getInstance($parentModuleModel, $relatedModuleModel);

        return $relatedList->getQuery($this);
    }

    /**
     * @throws Exception
     */
    public function getRelatedContactId()
    {
        $adb = PearDatabase::getInstance();
        $sql = $this->getRelationQuery('Contacts');
        $result = $adb->query($sql);

        return $adb->query_result($result, 0, 'crmid');
    }

    /**
     * @return false|Vtiger_Record_Model
     * @throws Exception
     */
    public function getRelatedContact()
    {
        $recordId = $this->getRelatedContactId();

        if (!empty($recordId)) {
            return Vtiger_Record_Model::getInstanceById($recordId, 'Contacts');
        }

        return false;
    }

    /**
     * @throws Exception
     */
    public function getRelatedContactName()
    {
        $contact = $this->getRelatedContact();

        return $contact ? $contact->getName() : '';
    }

    /**
     * @throws Exception
     */
    public function getEmailFromContacts()
    {
        $recordModel = $this->getRelatedContact();

        if (!empty($recordModel)) {
            return ITS4YouSignature_Module_Model::getEmailFromRecord($recordModel);
        }

        return '';
    }

    /**
     * @return string
     * @throws Exception
     */
    public function getEmailFromEmails()
    {
        $adb = PearDatabase::getInstance();
        $sql = $this->getRelationQuery('Emails');
        $result = $adb->query($sql);
        $recordId = $adb->query_result($result, 0, 'crmid');

        if (!empty($recordId)) {
            $recordModel = Vtiger_Record_Model::getInstanceById($recordId, 'Emails');

            if ($recordModel) {
                $emails = htmlspecialchars_decode($recordModel->get('saved_toid'));

                return trim($emails, '"[]');
            }
        }

        return '';
    }

    /**
     * @throws Exception
     */
    public function assignCopyToSignPerson()
    {
        $email = $this->getEmailFromEmails();

        if (empty($email)) {
            $email = $this->getEmailFromContacts();
        }

        $this->set('ccmail', $email);
    }

    /**
     * @return mixed
     * @throws Exception
     */
    public function sendEmail()
    {
        $recipientId = (int)$this->get('recipientId');
        $recipientEmail = $this->get('recipientEmail');
        $email = ITS4YouSignature_Emails_Model::getCleanInstance('Emails');
        $email->set('subject', $this->get('emailSubject'));
        $email->set('description', $this->getEmailMessage());
        $email->set('toemailinfo', [
            $recipientId => [$recipientEmail]
        ]);
        $email->set('saved_toid', $recipientEmail);
        $email->set('parent_id', $this->getRelatedContactId());
        $email->set('signature_id', $this->getId());
        $email->set('email_flag', 'SENT');
        $email->set('assigned_user_id', $this->get('assigned_user_id'));

        if (!$this->isEmpty('ccmail')) {
            $email->set('ccmail', $this->get('ccmail'));
        }

        $email->save();

        $this->setRelation($email->getId(), $email->getModuleName());

        return $email->send();
    }

    /**
     * @param string $value
     */
    public function setStatus($value)
    {
        $this->set('mode', 'edit');
        $this->set('signature_status', $value);
        $this->save();
    }

    /**
     * @return bool
     */
    public function isSigned()
    {
        return 'Signed' === $this->getStatus();
    }

    public function isWaitingForConfirmation()
    {
        return 'Waiting for confirmation' === $this->getStatus();
    }

    /**
     * @return string
     */
    public function getStatus()
    {
        return (string)$this->get('signature_status');
    }

    /**
     * @return bool
     */
    public function isCompleted()
    {
        return in_array($this->getStatus(), ['Completed', 'Signed', 'Signed and Confirmed']);
    }

    public function getDocumentsQuery()
    {
        return 'SELECT vtiger_notes.notesid AS record, vtiger_crmentity.setype AS module FROM vtiger_notes
				INNER JOIN vtiger_senotesrel ON vtiger_senotesrel.notesid= vtiger_notes.notesid
				LEFT JOIN vtiger_notescf ON vtiger_notescf.notesid= vtiger_notes.notesid
				INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid= vtiger_notes.notesid AND vtiger_crmentity.deleted=0
				INNER JOIN vtiger_crmentity AS crm2 ON crm2.crmid=vtiger_senotesrel.crmid
				LEFT JOIN vtiger_groups ON vtiger_groups.groupid = vtiger_crmentity.smownerid
				LEFT JOIN vtiger_seattachmentsrel ON vtiger_seattachmentsrel.crmid =vtiger_notes.notesid
				LEFT JOIN vtiger_attachments ON vtiger_seattachmentsrel.attachmentsid = vtiger_attachments.attachmentsid
				LEFT JOIN vtiger_users ON vtiger_crmentity.smownerid= vtiger_users.id
				WHERE crm2.crmid=?';
    }

    public function deleteDocuments()
    {
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery($this->getDocumentsQuery(), array($this->getId()));

        while ($row = $adb->fetchByAssoc($result)) {
            $document = Vtiger_Record_Model::getInstanceById($row['record'], 'Documents');

            if ($document) {
                $document->delete();
            }
        }
    }

    /**
     * @var string
     */
    public $emailTemplate = 'templates/DefaultSign.tpl';

    /**
     * @param string $value
     */
    public function setEmailTemplate($value) {
        $this->emailTemplate = $value;
    }

    /**
     * @return string
     */
    public function getEmailTemplate()
    {
        $templates = [
            'templates/DefaultSign.tpl' => 'templates/Sign.tpl',
            'templates/DefaultSigned.tpl' => 'templates/Signed.tpl',
        ];
        $url = 'layouts/' . Vtiger_Viewer::getDefaultLayoutName() . '/modules/' . (string) $this->getModuleName() . '/' . $templates[$this->emailTemplate];

        if(isset($templates[$this->emailTemplate]) && is_file($url)) {
            $this->emailTemplate = $templates[$this->emailTemplate];
        }

        return $this->emailTemplate;
    }

    /**
     * @return array
     */
    public function getEmailMessageVariables()
    {
        return array(
            'SIGNATURE_TITLE' => $this->get('emailSubject'),
            'SIGNATURE_MESSAGE' => $this->get('emailMessage'),
            'SIGNATURE_URL' => $this->getSignatureUrl(),
        );
    }

    /**
     * @return string
     * @throws Exception
     */
    public function getEmailMessage()
    {
        return !$this->isEmpty('emailBody') ? $this->getEmailBodyContent() : $this->getEmailTemplateContent();
    }

    /**
     * @return string
     */
    public function getEmailBodyContent()
    {
        $content = $this->get('emailBody');

        foreach ($this->getEmailMessageVariables() as $key => $value) {
            $content = str_replace('$' . $key . '$', $value, $content);
        }

        return (string)$content;
    }

    /**
     * @return string
     * @throws Exception
     */
    public function getEmailTemplateContent()
    {
        $viewer = new Vtiger_Viewer();
        $viewer->assign('MODULE', $this->getModuleName());
        $viewer->assign('RECORD_MODEL', $this);
        $viewer->assign('RECIPIENT_RECORD', $this->getRelatedContact());
        $viewer->assign('COMPANY', Vtiger_CompanyDetails_Model::getInstanceById());
        $viewer->assign('COMPANY_LOGO', $this->getCompanyImage());
        $viewer->assign('CURRENT_USER_MODEL', Users_Record_Model::getCurrentUserModel());

        foreach ($this->getEmailMessageVariables() as $key => $value) {
            $viewer->assign($key, $value);
        }

        return (string)$viewer->view($this->getEmailTemplate(), $this->getModuleName(), true);
    }

    public function getCompanyImage() {
        $company = Vtiger_CompanyDetails_Model::getInstanceById();
        $logo = $company->getLogo();

        return $logo->get('imagepath');
    }

    /**
     * @return string
     */
    public function getSignatureUrl()
    {
        $site_URL = ITS4YouSignature_Module_Model::getSiteUrl();

        return $site_URL . 'ITS4YouSignature.php?u=' . md5($site_URL) . '&s=' . base64_encode($this->getId());
    }

    /**
     * @param array $values
     */
    public function updateValues($values)
    {
        $adb = PearDatabase::getInstance();

        foreach ($values as $key => $value) {
            $sql = sprintf('UPDATE its4you_signature SET %s = ? WHERE signatureid = ?', $key);
            $adb->pquery($sql, [$value, $this->getId()]);
        }
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getAttachments()
    {
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery($this->getDocumentsQuery(), array($this->getId()));
        $attachments = array();

        while ($row = $adb->fetchByAssoc($result)) {
            $document = Vtiger_Record_Model::getInstanceById($row['record'], 'Documents');

            if ($document) {
                $documentDetails = $document->getFileDetails();

                if (empty($documentDetails['storedname'])) {
                    $documentDetails['storedname'] = $documentDetails['name'];
                }

                $file = $documentDetails['path'] . $documentDetails['attachmentsid'] . '_' . $documentDetails['storedname'];

                $attachments[] = [
                    'fileid' => $documentDetails['attachmentsid'],
                    'attachment' => decode_html($documentDetails['name']),
                    'path' => $documentDetails['path'],
                    'file' => $file,
                    'size' => filesize($file),
                    'type' => $documentDetails['type'],
                    'cid' => $documentDetails['cid'],
                ];
            }
        }

        return $attachments;
    }

    /**
     * @return bool
     */
    public function saveTemplate()
    {
        $this->updateValues([
            'template' => $this->get('templateId'),
            'template_body' => $this->get('templateBody'),
            'language' => $this->get('templateLanguage'),
            'source_record' => $this->get('sourceRecord'),
            'source_module' => $this->get('sourceModule'),
            'signature_mode' => $this->get('signatureMode'),
        ]);

        return true;
    }

    /**
     * @return string
     */
    public function getCurrentUserDate()
    {
        return (new DateTimeField(Vtiger_Functions::currentUserDisplayDateNew()))->getDBInsertDateValue();
    }

    public function saveSentOn()
    {
        $this->set('mode', 'edit');
        $this->set('signature_status', 'Waiting for Others');
        $this->set('senton', $this->getCurrentUserDate());
        $this->save();
    }

    public function saveSignatureDate()
    {
        $this->set('mode', 'edit');
        $this->set('signature_status', 'Waiting for confirmation');
        $this->set('signature_date', $this->getCurrentUserDate());
        $this->save();
    }

    /**
     * @param Vtiger_Request $request
     * @return ITS4YouSignature_Record_Model
     */
    public static function getInstanceFromRequest(Vtiger_Request $request)
    {
        /** @var $record ITS4YouSignature_Record_Model */
        $record = self::getCleanInstance('ITS4YouSignature');
        $record->set('signature_name', $request->get('emailSubject'));
        $record->set('signature_status', 'Created');
        $record->save();

        return $record;
    }

    /**
     * @return Vtiger_Record_Model
     */
    public function getUser()
    {
        return Users_Record_Model::getInstanceById($this->get('assigned_user_id'), 'Users');
    }

    /**
     * @return false|Vtiger_Record_Model
     */
    public function getSourceRecord()
    {
        if ($this->isEmpty('source_record') || !isRecordExists(intval($this->get('source_record')))) {
            return false;
        }

        return Vtiger_Record_Model::getInstanceById($this->get('source_record'));
    }

    /**
     * @return string
     */
    public function getSourceRecordName()
    {
        $sourceRecord = $this->getSourceRecord();

        return $sourceRecord ? $sourceRecord->getName() : '';
    }

    protected $settings;

    /**
     * @return Settings_Vtiger_Module_Model
     */
    public function getSettings()
    {
        if (!$this->settings) {
            $this->settings = Settings_Vtiger_Record_Model::getInstance('Settings:ITS4YouSignature');
        }

        return $this->settings;
    }

    /**
     * @throws Exception
     */
    public function getNumberField($module)
    {
        $tabId = getTabid($module);
        $adb = PearDatabase::getInstance();
        $result = $adb->pquery('SELECT fieldname FROM vtiger_field WHERE tabid = ? AND uitype = ?', array($tabId, 4));

        return $adb->query_result($result, 0, 'fieldname');
    }

    /**
     * @return string
     */
    public function getSignedSubject()
    {
        return vtranslate($this->getSettings()->get('signed_email_subject'), 'ITS4YouSignature') . ' - ' . $this->get('signature_name');
    }

    /**
     * @return string
     */
    public function getSignedMessage()
    {
        return (string)$this->getSettings()->get('signed_email_message');
    }

    /**
     * @throws Exception
     */
    public function sendSignedEmail()
    {
        $user = $this->getUser();

        if ($user) {
            $this->set('recipientId', $user->getId());
            $this->set('recipientEmail', $user->get('email1'));
            $this->set('emailSubject', $this->getSignedSubject());
            $this->set('emailMessage', $this->getSignedMessage());
            $this->setEmailTemplate('templates/DefaultSigned.tpl');
            $this->assignCopyToSignPerson();
            $this->sendEmail();
        }
    }

    public function assignPreContent(&$request)
    {
        if (!$this->isEmpty('template_body')) {
            $templateId = (int)$this->get('template');
            $language = $this->get('language');

            $sourceModule = $this->get('source_module');
            $sourceRecord = $this->get('source_record');

            $focus = CRMEntity::getInstance($sourceModule);
            $focus->retrieve_entity_info($sourceRecord, $sourceModule);

            $PDFContent = PDFMaker_PDFContent_Model::getInstance($templateId, $sourceModule, $focus, $language);
            $content = $PDFContent->getContent();

            $templateBody = decode_html($this->get('template_body'));
            $templateBody = str_replace('$PDF_SIGNATURE$', $PDFContent->getSignature(), $templateBody);

            $request['header' . $templateId] = $content['header'];
            $request['body' . $templateId] = $templateBody;
            $request['footer' . $templateId] = $content['footer'];
            $request['mode'] = 'edit';
        }
    }
}
