<?php
/**
 * XML Parser
 * Deze recursieve xml parser is gebaseerd op de in PHP ingebouwde maar incomplete xml parser
 *
 * @author W3Vision: Daniël van der Mark <daniel@w3vision.nl>
 * @copyright Er is geen copyright op dit bestand.
 *   Vicus is er mee bekend dat dit bestand door W3Vision is ontworpen,
 *   en nog steeds door W3Vision wordt gebruikt voor haar eigen producten
 */


/** Parse een string naar een tXML_Node (eventueel met subnodes)
 * Maak een tXML_Node aan en parse daarin de XML tekst in $XMLText
 * @param string $XMLText XML tekst
 * @param string $RootNodeName Naam van de root XML_Node
 * @param string $Encoding Naam van het encoding type, default = 'UTF-8'
 * @param boolean $CaseFolding Maak alles uppercase of niet
 * @return tXML_Node
 */
function Text2XML($XMLText, $RootNodeName = 'root', $Encoding = 'UTF-8', $CaseFolding = false)
{
  global $Dummy;
  $Result = new tXML_Node($Dummy, $RootNodeName);
  $Result->ParseText($XMLText, $Encoding, $CaseFolding);
  return $Result;
}

/** Parse een tekstbestand naar een tXML_Node (eventueel met subnodes)
 * Maak een tXML_Node aan en parse daarin de XML tekst in $XMLText
 * @param string $XMLFile XML file bestandsnaam
 * @param string $Encoding Naam van het encoding type, default = 'UTF-8'
 * @param boolean $CaseFolding Maak alles uppercase of niet
 * @return tXML_Node
 */
function File2XML($XMLFile, $Encoding = 'UTF-8', $CaseFolding = false)
{
  global $Dummy;

  $arrSettingXML = file($XMLFile);
  $XMLText = '';
  foreach ($arrSettingXML as $Row)
    $XMLText .= $Row;

  $Result = new tXML_Node($Dummy, 'root');
  $Result->ParseText($XMLText, $Encoding, $CaseFolding);
  return $Result->FirstChild();
}

/** Basis XML node voor tXML_TextNode en tXML_Node
 */
abstract class tXML_Base
{
  /** @var tXML_Node $Owner */
  public $Owner;
  public $AddLineFeed = false;

  /**
   * @param $Owner
   */
  function __construct(&$Owner)
  {
    $this->Owner = &$Owner;
  }

  /** Zoek indexnummer bij parent van deze node
   * @return integer Indexnummer (-1 wanneer niet gevonden)
   */
  public function IndexNr()
  {
    $i = false;
    if (isset($this->Owner->ChildNodes))
      for ($i = 0; $i < $this->Owner->ChildCount(); $i++)
        if ($this->Owner->ChildNodes[$i] == $this)
          break;
    return $i;
  }

  /**
   * @return mixed
   */
  abstract function GetXML();

  /**
   * @param string $Indent
   * @param string $LineFeed
   * @return mixed
   */
  abstract function GetXMLFormated($Indent = '', $LineFeed = "<br />\n");

/** Geef aantal ChildNodes van deze XML node terug
 * @return integer Aantal ChildNodes
 */
  public function ChildCount()
  {
    return 0;
  }

/** Heeft deze XML node ChildNodes?
 * @return boolean true=JA false=Nee
 */
  public function HasChildNodes()
  {
    return false;
  }
}


//--- Text xml node. For cData in XML. Used by tXML_Node ---------------------//
/** XML Tekst node
 * Hierin wordt de cData van een tXML_Node opgeslagen. Wordt (o.a.) aangemaakt door tXML_Node
 */
class tXML_TextNode extends tXML_Base
{
  public $Data;

  /**
   * @param $Owner
   * @param $Data
   */
  function __construct(&$Owner, $Data) // override
  {
    parent::__construct($Owner);
    $this->Data = $Data;
    $this->NodeName = 'XMLTextNode';
  }

  /**
   * @return mixed
   */
  public function GetXML() // override
  {
    return UTF8toXML($this->Data);
  }

  /**
   * @param string $Indent
   * @param string $LineFeed
   * @return string
   */
  public function GetXMLFormated($Indent = '', $LineFeed = "<br />\n") // override
  {
    return "\n<br />$Indent" . $this->Data;
  }
}

/** XML node en parser
 */
class tXML_Node extends tXML_Base
{
  public $Attributes = array();
  public $ChildNodes = array();
  public $OutputOuterXML = false;

  /** @var tXML_Node $ParserNode */
  public $ParserNode;
  public $Parser;
  public $NodeName;

  /**
   * @param $Owner
   * @param string $NodeName
   */
  function __construct(&$Owner, $NodeName = '') // override
  {
    parent::__construct($Owner);
    $this->NodeName = $NodeName;
  }

/** Geef attribute terug als deze bestaat. Geef anders een default waarde terug
 * @param string $AttributeName Naam van het attribute
 * @param string $DefaultValue Default waarde die wordt teruggegeven als het attribute niet bestaat
 * @return string Attribute waarde
 */
  public function AttrIfSet($AttributeName, $DefaultValue='')
  {
    return isset($this->Attributes[$AttributeName]) ? $this->Attributes[$AttributeName] : $DefaultValue;
  }

  /** Voeg een nieuwe ChildNode toe aan deze XML node
   * @param string $NodeName Naam van de nieuwe node
   * @param array $Attributes Array met 1 of meerdere attributen ("AttributeNaam"="Waarde")
   * @param string $Text [optioneel] (Wanneer niet leeg) voeg cData (via een tXML_TextNode) toe aan de nieuwe node
   * @param bool|string $Insert [optioneel] true = Nieuwe node bovenaan invoegen
   * @return tXML_Node De nieuwe XML node
   */
  public function &AddChildNode($NodeName, $Attributes = array(), $Text='', $Insert = false)
  {
    $Node = new tXML_Node($this, $NodeName);
    foreach ($Attributes as $Name => $Value)
      $Node->Attributes[$Name] = $Value;
    if ($Text != '')
      $Node->AddTextNode($Text);

    if ($Insert)
      array_unshift($this->ChildNodes, $Node);
    else
      $this->ChildNodes[] = $Node;
    return $Node;
  }

  /** Voeg cData toe aan deze node (via een tXML_TextNode)
   * @param string $Text De cData tekst
   * @param bool|string $Insert [optioneel] true = Tekst (tXML_TextNode) bovenaan invoegen
   * @internal param string $NodeName Naam van de nieuwe node
   * @return tXML_TextNode De tXML_TextNode van de cData
   */
  public function &AddTextNode($Text, $Insert = false)
  {
    if ($Insert)
      array_unshift($this->ChildNodes, new tXML_TextNode($this, $Text));
    else
  	  $this->ChildNodes[] = new tXML_TextNode($this, $Text);
    return $this->ChildNodes[count($this->ChildNodes) - 1];
    // TODO: De return werkt niet als Insert=true.
  }

/** Heeft deze XML node ChildNodes?
 * @return boolean true=JA false=Nee
 */
  public function HasChildNodes()
  {
    return count($this->ChildNodes) > 0;
  }

/** Geef aantal ChildNodes van deze XML node terug
 * @return integer Aantal ChildNodes
 */
  public function ChildCount()
  {
    return count($this->ChildNodes);
  }

/** Verwijder een node
 * @param tXML_Base $Node - XML node of tekst node object
 */
  public function DeleteNode(&$Node)
  {
    $Index = $Node->IndexNr();
    //$Node->Clear();
    unset($this->ChildNodes[$Index]);
  }

/** Haal eerste childnode op
 * @return tXML_Base XML node of tekst node object
 */
  public function FirstChild()
  {
    return $this->ChildNodes[0];
  }

  /**
   * @param string $NodeName
   * @return bool
   */
  public function FindNode($NodeName)
  {
    $Dummy = null;
    return $this->FindNextNode($NodeName, $Dummy);
  }

  /**
   * @param string $NodeName
   * @param tXML_Base|null $PreviousNode
   * @return bool
   */
  public function &FindNextNode($NodeName, &$PreviousNode)
  {
    $Result = false;
    $PreviousNodeFound = ($PreviousNode === null);
    for ($i = 0; $i < count($this->ChildNodes); $i++)
      if ($PreviousNodeFound)
      {
        if ($this->ChildNodes[$i]->NodeName == $NodeName)
        {
          $Result = &$this->ChildNodes[$i];
          break 1;  // Breaks out of the 'for' loop
        }
      }
      else
        if ($this->ChildNodes[$i] === $PreviousNode)
          $PreviousNodeFound = true;
    return $Result;
  }

  /**
   * @param string $NodeName
   * @param $AttributeName
   * @param string $AttributeValue
   * @param bool $Recursive
   *
   * @internal param \tXML_Node $Node
   * @return bool
   */
  public function FindNodeWithAttribute($NodeName, $AttributeName, $AttributeValue, $Recursive = false)
  {
    $Result = false;
    /** @var $Node tXML_Node */
    foreach ($this->ChildNodes as $Node)
    {
      if ($Node->NodeName == $NodeName)
        if (isset($Node->Attributes[$AttributeName]))
          if ($Node->Attributes[$AttributeName] == $AttributeValue)
          {
            $Result = $Node;
            break 1;
          }
      if ($Recursive)
      {
        /** @var $Result tXML_Node */
        $Result = $Node->FindNodeWithAttribute($NodeName, $AttributeName, $AttributeValue, true);
        if (isset($Result->Attributes))
          break 1;
      }
    }
    return $Result;
  }

  /**
   * @param bool $My_AddLineFeed
   * @param string $Indent
   * @return string
   */
  public function GetXML($My_AddLineFeed = false, $Indent = '') // override
  {
    $AddLineFeed = ($My_AddLineFeed) ? true : $this->AddLineFeed;
    $LF = $AddLineFeed ? "\n" : '';
    $IndAdd = $AddLineFeed ? '  ' : '';
    $InnerIndent = $AddLineFeed ? $Indent : '';

    $Result = "$InnerIndent" . $this->GetOuterXML(count($this->ChildNodes) < 1) . "$LF";
    if (count($this->ChildNodes) > 0)
    {
      $Result .= $this->GetInnerXML($AddLineFeed, $Indent . $IndAdd);
      $Result .= "$InnerIndent</" . $this->NodeName . ">$LF";
    }
    return $Result;
  }

  /**
   * @param bool $ClosedNode
   * @return string
   */
  public function GetOuterXML($ClosedNode = false)
  {
  	$Result = "<"  . $this->NodeName;
    foreach ($this->Attributes as $key => $value)
      $Result .= " $key=\"" . UTF8toXML($value) . "\"";
  	$Result .= $ClosedNode ? " />" : ">";
  	return $Result;
  }

  /**
   * @param bool $My_AddLineFeed
   * @param string $Indent
   * @return string
   */
  public function GetInnerXML($My_AddLineFeed = false, $Indent = '')
  {
    $AddLineFeed = ($My_AddLineFeed) ? true : $this->AddLineFeed;
    $IndAdd = $AddLineFeed ? '  ' : '';
    $Result = '';
    /** @var $Node tXML_Base */
    foreach ($this->ChildNodes as $Node)
      $Result .= $Node->GetXML($AddLineFeed, $IndAdd . $Indent);
    return $Result;
  }

  /**
   * @param string $NodeName
   * @param bool $My_AddLineFeed
   * @param string $Indent
   * @return string
   */
  public function GetCloseNode($NodeName = '', $My_AddLineFeed = false, $Indent = '')
  {
    $DisplayName = ($NodeName == '') ? $this->NodeName : $NodeName;
    $AddLineFeed = ($My_AddLineFeed) ? true : $this->AddLineFeed;
    $LF = $AddLineFeed ? "\n" : '';
    $InnerIndent = $AddLineFeed ? $Indent : '';
    $Result = "{$LF}{$InnerIndent}</{$DisplayName}>";
    return $Result;
  }

  /**
   * @param string $Indent
   * @param string $LineFeed
   * @return string
   */
  public function GetXMLFormated($Indent = '', $LineFeed = "<br />\n") // override
  {
    $Result = "\n<br />$Indent&lt;" . $this->NodeName;
    foreach ($this->Attributes as $key => $value)
      $Result .= ($value != null) ? " $key=\"" . UTF8toXML($value) . "\"" : " $key";
    if (count($this->ChildNodes) > 0)
    {
      $Result .= "&gt;";
      /** @var $Node tXML_Node */
      foreach ($this->ChildNodes as $Node)
        $Result .= $Node->GetXMLFormated("$Indent&nbsp;&nbsp;");
      $Result .= "\n<br />$Indent&lt;/" . $this->NodeName . "&gt;";
    }
    else
      $Result .= " /&gt;";
    return $Result;
  }

  //--- Native PHP XMLParser functions ---------------------------------------//
  /**
   * @param $XMLText
   * @param string $Encoding
   * @param bool $CaseFolding
   * @return int
   * @throws tSpeakap_Exception
   */
  public function &ParseText($XMLText, $Encoding = 'UTF-8', $CaseFolding = false)
  {
    $this->ParserNode =& $this;
    $this->Parser = xml_parser_create($Encoding);
    xml_set_object($this->Parser, $this);
    xml_parser_set_option($this->Parser, XML_OPTION_CASE_FOLDING, ($CaseFolding ? 1 : 0));
    xml_set_element_handler($this->Parser, "Parse_OpenTag", "Parse_CloseTag");
    xml_set_character_data_handler($this->Parser, "Parse_CData");
    $Result = xml_parse($this->Parser, $XMLText, true);
    if (!$Result)
    {
      $Error = "XML Parser error: [" . xml_get_error_code($this->Parser) . "] \"" .
        xml_error_string(xml_get_error_code($this->Parser)) .
        "\" on line: " . xml_get_current_line_number($this->Parser) .
        " character: " . xml_get_current_column_number($this->Parser);
      throw new tSpeakap_Exception($Error);
      //echo "\n\n$Error <br /><br />\n\n";
    }
    xml_parser_free($this->Parser);
    return $Result;
  }

  /**
   * @param $Parser
   * @param $Tag
   * @param $Attributes
   */
  public function Parse_OpenTag(&$Parser, $Tag, $Attributes)
  {
    $Node = (!$this->ParserNode) ? $this->ParserNode : $this;
    $NewNode =& $Node->AddChildNode($Tag);
    foreach ($Attributes as $Key => $Value)
      $NewNode->Attributes[$Key] = $Value;
    $this->ParserNode =& $NewNode;
  }

  /**
   * @param $Parser
   * @param $CData
   */
  public function Parse_CData(&$Parser, $CData)
  {
    $Node =& $this->ParserNode;
    if (trim($CData) != '')
      $Node->AddTextNode($CData);
  }

  /**
   * @param $Parser
   * @param $Tag
   */
  public function Parse_CloseTag(&$Parser, $Tag)
  {
    $this->ParserNode =& $this->ParserNode->Owner;
  }

  /**
   * @param array $Array
   * @param string $Fields
   */
  public function Add_CData_Array(&$Array, $Fields = null)
  {
    $FieldArr = $Fields === null ? array() : TrimExplode(',', $Fields);
    foreach ($Array as $Key => $Value)
    {
      if ((count($FieldArr) == 0) || in_array($Key, $FieldArr))
        $this->AddChildNode($Key, array(), $Value);
    }
  }
}

/**
 * Add_XML_Array()
 *
 * @param tXML_Node $ParentNode
 * @param string $NodeName
 * @param array $Array
 * @param string Fields
 * @return bool
 */
function Add_XML_Array(&$ParentNode, $NodeName, &$Array, $Fields = null)
{
  $ArrayNode = $ParentNode->AddChildNode($NodeName);
  $FieldArr = $Fields === null ? array() : TrimExplode(',', $Fields);
  foreach ($Array as $Key => $Value)
  {
    if ((count($FieldArr) == 0) || in_array($Key, $FieldArr))
      $ArrayNode->AddChildNode($Key, array(), $Value);
  }
}
