<?php
/**
 * tMultiQuery
 *
 * @package 000-Speakap
 * @subpackage Core_DB
 * @version 2.0
 * @author daniel.mark@speakap.nl
 */

class tMultiQuery
{
  /** @var mysqli $Connection */
  public $Connection = false;
  public $DB = false;
  public $Results = array();
  public $SQL = array();
  public $VarList = array();

  /**
   * tMultiQuery::__construct()
   *
   * @param mysqli $Connection
   * @param tDB $DB
   * @param string $SQL
   * @return \tMultiQuery
   */
  public function __construct($Connection, $DB, $SQL = '')
  {
    $this->Connection = $Connection;
    $this->DB = $DB;
    if ($SQL != '')
      $this->SQL[] = $SQL;
  }

  /**
   * tMultiQuery::ClearAll() - Clear all Multiquery's SQL, Var's and Results
   *
   * @return void
   */
  public function Clear()
  {
    $this->Results = array();
    $this->SQL = array();
    $this->VarList = array();
  }

  /**
   * tMultiQuery::Escape()
   *
   * @param string $Value
   * @return string
   */
  public function Escape($Value)
  {
    return $this->Connection->escape_string($Value);
  }

  /**
   * tMultiQuery::Add_Set()
   *
   * @param string $VarName
   * @param bool|string $ValueSQL = false to only add the var to the "Do not escape" list
   *   - Works like sprintf @see http://php.net/manual/en/function.sprintf.php
   * @internal param mixed $Arg1 ..n
   * @return string
   */
  public function Add_Set($VarName, $ValueSQL = false /*, Arg1, Arg2 ...*/)
  {
    xArray::AddUnique($this->VarList, $VarName);
    if ($ValueSQL === false)
      return false;

    // Escape ValueSQL string
    $Arguments = func_get_args();
    $SQLArgs = array($ValueSQL);
    foreach ($Arguments as $Key => &$Arg)
    if ($Key > 1)
      $SQLArgs[] = ($Arg === null) ? 'null' : $this->Escape($Arg);
    $ValueSQL = call_user_func_array('sprintf', $SQLArgs);

    return $this->SQL[] = "SET $VarName := $ValueSQL;";
  }

  /**
   * tMultiQuery::Add_SQL()
   *
   * @internal param string $SQL - Works like sprintf @see http://php.net/manual/en/function.sprintf.php
   * @internal param mixed $Arg1 ..n
   * @return string
   */
  public function Add_SQL(/* $SQL, Arg1, Arg2 ...*/)
  {
    $Arguments = func_get_args();
    foreach ($Arguments as $Key => &$Arg)
    if ($Key > 0)
      $Arg = ($Arg === null) ? 'null' : $this->Escape($Arg);
    $SQL = call_user_func_array('sprintf', $Arguments);

    return $this->SQL[] = $SQL;
  }

  /**
   * tMultiQuery::Add_Update()
   *
   * @param string $TableName
   * @param array $Fields
   * @param string $WhereSQL - Works like sprintf @see http://php.net/manual/en/function.sprintf.php
   * @throws tSQL_Exception
   * @internal param mixed $Arg1 ..n
   * @return string
   */
  public function Add_Update($TableName, $Fields, $WhereSQL /*, Arg1, Arg2 ...*/)
  {
    // Escape Where string
    $Arguments = func_get_args();
    $WhereArgs = array($WhereSQL);
    foreach ($Arguments as $Key => &$Arg)
    if ($Key > 2)
      $WhereArgs[] = ($Arg === null) ? 'null' : $this->Escape($Arg);
    $WhereSQL = call_user_func_array('sprintf', $WhereArgs);

    // Escape Fields
    $FieldTypes = $this->DB->Get_FieldTypes($TableName, true);
    foreach ($Fields as $Field => $Value)
      $Fields[$Field] = $this->Escape($Field) .  " = " . (in_array($Value, $this->VarList) ?
        $Value : $this->DB->Convert_4_SQL_Write($Value, $FieldTypes[$Field]));

    if (count($Fields) < 1)
      throw new tSQL_Exception(Err_SQL_Update__NoFieldsToUpdate, 'WHERE ' . $WhereSQL);
    $SQL = sprintf("UPDATE %s SET %s WHERE %s;", $TableName,
      implode(',', array_values($Fields)), $WhereSQL);

    return $this->SQL[] = $SQL;
  }

  /**
   * tMultiQuery::Add_Insert()
   *
   * @param string $TableName
   * @param array $Fields
   * @throws tSQL_Exception
   * @return string
   */
  public function Add_Insert($TableName, $Fields)
  {
    // Escape Fields
    $FieldTypes = $this->DB->Get_FieldTypes($TableName, true);
    foreach ($Fields as $Field => $Value)
      $Fields[$Field] = in_array($Value, $this->VarList) ? $Value :
        $this->DB->Convert_4_SQL_Write($Value, $FieldTypes[$Field]);

    if (count($Fields) < 1)
      throw new tSQL_Exception(Err_SQL_Insert__NoFieldsToInsert, '');
    $SQL = sprintf("INSERT INTO %s (%s) VALUES (%s);", $TableName, implode(', ', array_keys($Fields)),
      implode(', ', array_values($Fields)));

    return $this->SQL[] = $SQL;
  }

  /**
   * tMultiQuery::Add_Insert_Multi()
   *
   * @param string $TableName
   * @param array $Rows
   * @throws tSQL_Exception
   * @return string
   */
  public function Add_Insert_Multi($TableName, $Rows)
  {
    // Maximaal aantal insert rows
    $Fields = array();
    foreach ($Rows as $Row)
      foreach ($Row as $FieldName => $Field)
        xArray::AddUnique($Fields, $FieldName);
    $FieldStr = implode(', ', $Fields);
    $FieldTypes = $this->DB->Get_FieldTypes($TableName, true);
    $InsertSQL = "INSERT INTO $TableName ($FieldStr) VALUES ";
    $FirstRow = true;
    foreach ($Rows as $Row)
    {
      $InsertSQL .= $FirstRow ? '(' : ', (';
      $FirstField = true;
      if (count($Row) < 1)
        throw new tSQL_Exception(Err_SQL_MultiInsert__NoFieldsToInsert, $InsertSQL);

      foreach ($Fields as $Field)
      {
        $FieldValue = GetIfSet($Row, $Field, null);
        if (!in_array($FieldValue, $this->VarList))
          $FieldValue = $this->DB->Convert_4_SQL_Write($FieldValue, $FieldTypes[$Field]);
        $InsertSQL .= $FirstField ? $FieldValue : ", {$FieldValue}";
        $FirstField = false;
      }
      $InsertSQL .= ')';
      $FirstRow = false;
    }
    if (count($Rows) < 1)
      throw new tSQL_Exception(Err_SQL_MultiInsert__NoRecordsToInsert, $InsertSQL);

    $this->SQL[] = $InsertSQL . ';';
    return $InsertSQL;
  }

  /**
   * tMultiQuery::Add_Delete()
   *
   * @param string $TableName
   * @param string $WhereSQL - Works like sprintf @see http://php.net/manual/en/function.sprintf.php
   * @internal param mixed $Arg1 ..n
   * @return string SQL string to execute
   */
  public function Add_Delete($TableName, $WhereSQL /*, Arg1, Arg2 ...*/)
  {
    $Arguments = func_get_args();
    $WhereArgs = array($WhereSQL);
    foreach ($Arguments as $Key => &$Arg)
    if ($Key > 1)
      $WhereArgs[] = ($Arg === null) ? 'null' : $this->Escape($Arg);
    $WhereSQL = call_user_func_array('sprintf', $WhereArgs);

    return $this->SQL[] = "DELETE FROM {$TableName} WHERE {$WhereSQL};";
  }

  /**
   * tMultiQuery::Get_SQL()
   *
   * @param string $Seperator
   * @return string
   */
  public function Get_SQL($Seperator = "\n")
  {
    return implode($Seperator, $this->SQL);
  }


  public function Execute_Real($AddTransaction = false, $QueryToFirePHP = false, $FirePHP_Label = '')
  {
    global $Last_MySQL_Query, $SQL2FirePHP;
    $this->Results = array();
    $Last_MySQL_Query = $AddTransaction ? "START TRANSACTION; " . $this->Get_SQL(' ') .
      " COMMIT;" : $this->Get_SQL(' ');
    if ($SQL2FirePHP || $QueryToFirePHP)
      FB::send($Last_MySQL_Query, 'tMultiQuery::Execute - ' . $FirePHP_Label);
    if ($this->Connection->real_query($Last_MySQL_Query))
    {
      do
      {
        /** @var $Result mysqli_result */
        if ($Result = $this->Connection->use_result())
        {
          $this->Results[] = new tQueryResult($Result);
          $Result->free();
        }
      }
      while ($this->Connection->more_results() && $this->Connection->next_result());
    }
    if ($this->Connection->error)
      throw new tSQL_Exception(Err_SQL_Error);
    $Last_MySQL_Query = '';
    return $this->Results;
  }

  /**
   * tMultiQuery::Execute()
   *
   * @param bool $AddTransaction
   * @param bool $QueryToFirePHP
   * @param string $FirePHP_Label
   * @throws tSQL_Exception
   * @return array
   */
  public function Execute($AddTransaction = false, $QueryToFirePHP = false, $FirePHP_Label = '')
  {
    global $Last_MySQL_Query, $SQL2FirePHP, $Last_MySQL_Error;
    $this->Results = array();
    $Last_MySQL_Query = $AddTransaction ? "START TRANSACTION; " . $this->Get_SQL(' ') .
      " COMMIT;" : $this->Get_SQL(' ');
    if ($SQL2FirePHP || $QueryToFirePHP)
      FB::send($Last_MySQL_Query, 'tMultiQuery::Execute - ' . $FirePHP_Label);
    if ($this->Connection->multi_query($Last_MySQL_Query))
    {
      do
      {
        /** @var $Result mysqli_result */
        if ($Result = $this->Connection->use_result())
        {
          $this->Results[] = new tQueryResult($Result);
          $Result->free();
        }
      }
      while ($this->Connection->more_results() && $this->Connection->next_result());
    }
    $Last_MySQL_Error = $this->Connection->error;
    if ($this->Connection->error)
      throw new tSQL_Exception(Err_SQL_Error);
    $Last_MySQL_Query = '';
    return $this->Results;
  }

  /**
   * tMultiQuery::ResultByFieldName()
   *
   * @param string $FieldName
   * @param bool $OrgTableName
   * @return mixed tQueryResult
   */
  public function ResultByFieldName($FieldName, $OrgTableName = false)
  {
    $Result = false;
    /** @var tQueryResult $QResult */
    if (is_array($this->Results))
      foreach ($this->Results as $QResult)
        if ($QResult->FieldByName($FieldName, $OrgTableName) !== false)
          $Result = $QResult;
    return $Result;
  }

  public function Result_Rows_ByFieldName($FieldName, $OrgTableName = false)
  {
    $Result = $this->ResultByFieldName($FieldName, $OrgTableName);
    if ($Result)
      return $Result->Rows;
    else
      throw new tSQL_Exception(Err_SQL_Error);
  }

  public function Result_1stRow_ByFieldName($FieldName, $OrgTableName = false)
  {
    $Result = $this->Result_Rows_ByFieldName($FieldName, $OrgTableName);
    return xArray::Get_Array($Result, 0, false, Err_SQL_No_Records_Found);
  }
}

/**
 * tQueryResult
 *
 * @package
 * @author Speakap B.V.
 * @copyright Speakap B.V.
 * @version 2012
 * @access public
 */
class tQueryResult
{
  public $Fields = array();
  public $Rows = array();

  /**
   * tQueryResult::__construct()
   *
   * @param mysqli_result $Result
   * @return \tQueryResult
   */
  public function __construct($Result)
  {
    $this->Rows = array();
    $Fields = $Result->fetch_fields();
    foreach ($Fields as $Field)
      $this->Fields[$Field->name] = $Field;

    while ($Row = $Result->fetch_array(MYSQLI_ASSOC))
      $this->Rows[] = $Row;
  }

  /**
   * tQueryResult::Row_Count()
   *
   * @return Integer - Number of rows in result
   */
  public function Row_Count()
  {
    return count($this->Rows);
  }

  /**
   * tQueryResult::FieldByName()
   *
   * @param string $FieldName
   * @param bool $OrgTableName
   * @return bool/object - {name: ,orgname: ,table: ,def: ,db: ,catalog: ,maxlength: ,
   *   length: ,charsetnr: ,flags: ,type: ,decimals: }
   */
  public function FieldByName($FieldName, $OrgTableName = false)
  {
    $Result = false;
    foreach ($this->Fields as $Field)
      if ($FieldName == $Field->name)
        if (($OrgTableName == false) || ($OrgTableName == $Field->orgtable))
        {
          $Result = $Field;
          break;
        }
    return $Result;
  }
}
