<?php
/*+***********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.0
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * All Rights Reserved.
 *************************************************************************************/

class Settings_VebMapping_Mapping_Model extends Settings_Vtiger_Module_Model {

	var $name = 'VebMapping';
	var $mapping;

	/**
	 * Function to get detail view url of this model
	 * @return <String> url
	 */
	public function getDetailViewUrl() {

		return sprintf('index.php?parent=%s&module=%s&view=%s%s',
					$this->getParentName(),
					$this->getName()
			);
	}

	/**
	 * Function to get edit view url of this model
	 * @return <String> url
	 */
	public function getEditViewUrl() {

		// return sprintf('index.php?parent=%s&module=%s&view=MappingEdit&src=%s&dst=%s',
		return sprintf('index.php?parent=%s&module=%s&view=MappingEdit',
					$this->getParentName(),
					$this->getName()
			);
	}

	/**
	 * Function to get delete url of this mapping model
	 * @return <String> url
	 */
	public function getMappingDeleteUrl() {

		return sprintf('index.php?parent=%s&module=%s&action=Delete',
					$this->getParentName(),
					$this->getName()
			);
	}

	/**
	 * Function to get list of detail view link models
	 * @return <Array> list of detail view link models <Vtiger_Link_Model>
	 */
	public function getDetailViewLinks() {

		return array(Vtiger_Link_Model::getInstanceFromValues(array(
				'linktype' => 'DETAILVIEW',
				'linklabel' => 'LBL_EDIT',
				'linkurl' => 'javascript:Settings_VebMapping_Js.triggerEdit("'. $this->getEditViewUrl() .'")',
				'linkicon' => ''
				)));
	}

	/**
	 * Function to get list of mapping link models
	 * @return <Array> list of mapping link models <Vtiger_Link_Model>
	 */
	public function getMappingLinks() {

		return array(Vtiger_Link_Model::getInstanceFromValues(array(
				'linktype' => 'DETAILVIEW',
				'linklabel' => 'LBL_DELETE',
				'linkurl' => 'javascript:Settings_VebMapping_Js.triggerDelete(event,"'. $this->getMappingDeleteUrl() .'")',
				'linkicon' => ''
				)));
	}

	/**
	 * Function to get mapping details
	 * @return <Array> list of mapping details
	 */
	public function getMapping($srcModule, $dstModule, $editable = 1) {

		$srcTabId = getTabId($srcModule);
		$dstTabId = getTabId($dstModule);

		$db = PearDatabase::getInstance();
		$query = 'SELECT * FROM vtiger_veb_mapping WHERE srctab = ? AND dsttab = ?';

		if ($editable == 1) $query .= ' AND editable = 1';
		if ($editable == 0) $query .= ' AND editable = 0';

		$params = array($srcTabId, $dstTabId);
		$result = $db->pquery($query, $params);
		$numOfRows = $db->num_rows($result);
		$mapping = array();

		for ($i=0; $i<$numOfRows; $i++) {

			$rowData = $db->query_result_rowdata($result, $i);
			$mapping[$rowData['cfmid']] = $rowData;
		}

		$finalMapping = $fieldIdsList = array();
		foreach ($mapping as $mappingDetails) {

			array_push($fieldIdsList, $mappingDetails['srcfid'], $mappingDetails['dstfid']);
		}

		$fieldLabelsList = array();

		if(!empty($fieldIdsList)){

			$fieldLabelsList = $this->getFieldsInfo(array_unique($fieldIdsList));
		}

		foreach ($mapping as $mappingId => $mappingDetails) {

			$finalMapping[$mappingId] = array(
								'editable'	=> $mappingDetails['editable'],
								'srcModule'	=> $fieldLabelsList[$mappingDetails['srcfid']],
								'dstModule'	=> $fieldLabelsList[$mappingDetails['dstfid']],
							);
		}

		return $finalMapping;
	}

	/**
	 * Function to get fields info
	 * @param <Array> list of field ids
	 * @return <Array> list of field info
	 */
	public function getFieldsInfo($fieldIdsList) {

		// $srcModel = Vtiger_Module_Model::getInstance($this->getName());
		// $srcId = $srcModel->getId();

		$db = PearDatabase::getInstance();
		$result = $db->pquery('SELECT fieldid, fieldlabel, uitype, typeofdata, fieldname, tablename, tabid
					FROM vtiger_field
					WHERE fieldid IN ('. generateQuestionMarks($fieldIdsList). ')', $fieldIdsList);

		$numOfRows = $db->num_rows($result);

		$fieldLabelsList = array();

		for ($i=0; $i<$numOfRows; $i++) {

			$rowData = $db->query_result_rowdata($result, $i);
			$fieldInfo = array('id' => $rowData['fieldid'], 'label' => $rowData['fieldlabel']);

			// if ($rowData['tabid'] === $srcId) {

				$fieldModel = Settings_Leads_Field_Model::getCleanInstance();
				$fieldModel->set('uitype', $rowData['uitype']);
				$fieldModel->set('typeofdata', $rowData['typeofdata']);
				$fieldModel->set('name', $rowData['fieldname']);
				$fieldModel->set('table', $rowData['tablename']);
				$fieldInfo['fieldDataType'] = $fieldModel->getFieldDataType();
			// }

			$fieldLabelsList[$rowData['fieldid']] = $fieldInfo;
		}
		return $fieldLabelsList;
	}

	/**
	 * Function to save the mapping info
	 * @param <Array> $mapping info
	 */
	public function save($mapping) {

		$db = PearDatabase::getInstance();
		$deleteMappingsList = $updateMappingsList = $createMappingsList = array();

		$srcTabId = $mapping['srctab'];
		$dstTabId = $mapping['dsttab'];

                // and unset
                unset($mapping['srctab']);
                unset($mapping['dsttab']);

		foreach ($mapping as $mappingDetails) {

			$mappingId = $mappingDetails['mappingid'];

			if ($mappingDetails['src']) {

				if ($mappingId) {

					if ((array_key_exists('deletable', $mappingDetails)) || !$mappingDetails['dst']) {

						$deleteMappingsList[] = $mappingId;
					}
					else {

						if ($mappingDetails['dst']) $updateMappingsList[] = $mappingDetails;
					}
				}
				else {

					if ($mappingDetails['dst']) $createMappingsList[] = $mappingDetails;
				}
			}
		}

		if ($deleteMappingsList) {

			$db->pquery('DELETE FROM vtiger_veb_mapping
					WHERE editable = 1
					AND cfmid IN ('. generateQuestionMarks($deleteMappingsList) .')', $deleteMappingsList);
		}

		if ($createMappingsList) {

			$insertQuery = 'INSERT INTO vtiger_veb_mapping (srctab, srcfid, dsttab, dstfid) VALUES ';

			$count = count($createMappingsList);

			$params = array();

			for ($i=0; $i<$count; $i++) {

				$mappingDetails = $createMappingsList[$i];
				$insertQuery .= '(?, ?, ?, ?)';
				array_push($params, $srcTabId, $mappingDetails['src'], $dstTabId, $mappingDetails['dst']);

				if ($i !== $count-1) {

					$insertQuery .= ', ';
				}
			}

			$db->pquery($insertQuery, $params);
		}

		if ($updateMappingsList) {

			$srcQuery	= ' SET srcfid = CASE ';
			$dstQuery	= ' dstfid = CASE ';
			$paramArray = $srcParamArray = $dstParamArray = array();

			foreach ($updateMappingsList as $mappingDetails) {

				$mappingId = $mappingDetails['mappingid'];

				$srcQuery	.= ' WHEN cfmid = ? THEN ? ';
				array_push($srcParamArray, $mappingId);
				array_push($srcParamArray, $mappingDetails['src']);

				$dstQuery	.= ' WHEN cfmid = ? THEN ? ';
				array_push($dstParamArray, $mappingId);
				array_push($dstParamArray, $mappingDetails['dst']);
			}

			$srcQuery	.= ' ELSE srcfid END ';
			$dstQuery	.= ' ELSE dstfid END ';

			$paramArray = array_merge($paramArray, $srcParamArray);
			$paramArray = array_merge($paramArray, $dstParamArray);

			array_push($paramArray, 1);

			$db->pquery("UPDATE vtiger_veb_mapping $srcQuery, $dstQuery
					WHERE editable = ?", $paramArray);
		}
	}

	/**
	 * Function to get restricted field ids list
	 * @return <Array> list of field ids
	 */
	public static function getRestrictedFieldIdsList() {

		$db = PearDatabase::getInstance();

		$result = $db->pquery('SELECT * FROM vtiger_veb_mapping
					WHERE editable = ?', array(0));

		$numOfRows = $db->num_rows($result);

		$restrictedIdsList = array();

		for ($i=0; $i<$numOfRows; $i++) {

			$rowData = $db->query_result_rowdata($result, $i);

			if ($rowData['srcfid']) {

				$restrictedIdsList[] = $rowData['srcfid'];
			}

			if ($rowData['dstfid']) {

				$restrictedIdsList[] = $rowData['dstfid'];
			}
		}

		return $restrictedIdsList;
	}

	/**
	 * Function to get mapping supported modules list
	 * @return <Array>
	 */
	public static function getSupportedModulesList() {

		return array('Accounts', 'Contacts', 'Potentials');
	}

	/**
	 * Function to get instance
	 * @param <Boolean> true/false
	 * @return <Settings_Leads_Mapping_Model>
	 */
	public static function getInstance($editable = false) {

		$instance = new self();
		// $instance->getMapping($editable);

		return $instance;
	}

	/**
	 * Function to get instance
	 * @return <Settings_Leads_Mapping_Model>
	 */
	public static function getCleanInstance() {

		return new self();
	}

	/**
	 * Function to delate the mapping
	 * @param <Array> $mappingIdsList
	 */
	public static function deleteMapping($mappingIdsList) {

		$db = PearDatabase::getInstance();
		$db->pquery('DELETE FROM vtiger_veb_mapping
				WHERE cfmid IN ('. generateQuestionMarks($mappingIdsList). ')', $mappingIdsList);
	}
}
