<?php
/*********************************************************************************
 * The content of this file is subject to the Quick Search 4 You license.
 * ("License"); You may not use this file except in compliance with the License
 * The Initial Developer of the Original Code is IT-Solutions4You s.r.o.
 * Portions created by IT-Solutions4You s.r.o. are Copyright(C) IT-Solutions4You s.r.o.
 * All Rights Reserved.
 ********************************************************************************/

require_once 'data/CRMEntity.php';
require_once 'modules/CustomView/CustomView.php';
require_once 'include/Webservices/Utils.php';
require_once 'include/Webservices/RelatedModuleMeta.php';

class ITS4YouQuickSearch_QueryGeneratorQS_Helper extends EnhancedQueryGenerator
{
    protected $meta;
    protected $searchValue;
    protected $conditionals;
    protected $groupInfo;
    protected $referenceFieldList;
    protected $whereFields;
    protected $ignoreComma;
    protected $query;
    protected $whereClause;
    protected $ownerFields;
    protected $conditionalWhere;

    public function __construct($module, $user)
    {
        parent::__construct($module, $user);

        $this->conditionals = array();
        $this->meta = $this->getMeta($module);
        $this->groupInfo = '';
        $this->whereFields = $this->getWhereFields();
        $this->ownerFields = $this->meta->getOwnerFields();
        $this->conditionalWhere = null;
    }

    public function startGroup($groupType)
    {
        $this->groupInfo .= " $groupType (";
    }

    public function endGroup()
    {
        $this->groupInfo .= ')';
    }

    public function addCondition($fieldname, $value, $operator, $glue = null, $newGroup = false, $newGroupType = null, $ignoreComma = false)
    {
        $conditionNumber = $this->conditionInstanceCount++;
        if ($glue != null && $conditionNumber > 0) {
            $this->addConditionGlue($glue);
        }

        $this->groupInfo .= "$conditionNumber";
        $this->whereFields[] = $fieldname;
        parent::addWhereField($fieldname);
        $this->ignoreComma = $ignoreComma;
        $this->reset();
        $this->conditionals[$conditionNumber] = $this->getConditionalArray($fieldname, $value, $operator);
    }

    protected function getConditionalArray($fieldname, $value, $operator)
    {
        if (is_string($value)) {
            $value = trim($value);
        } elseif (is_array($value)) {
            $value = array_map(trim, $value);
        }
        return array('name' => $fieldname, 'value' => $value, 'operator' => $operator);
    }

    /**
     * @throws Exception
     */
    public function getQuery()
    {
        if (!empty($this->query)) {
            return $this->query;
        }

        $allFields = array_merge($this->whereFields, $this->getFields());

        foreach ($allFields as $fieldName) {
            if (in_array($fieldName, $this->getReferenceFieldList())) {
                $refFieldInfoList = $this->getReferenceFieldInfoList();
                $moduleList = $refFieldInfoList[$fieldName];

                foreach ($moduleList as $module) {
                    if (empty($this->getModuleNameFields($module))) {
                        $this->getMeta($module);
                    }
                }
            } elseif (in_array($fieldName, $this->getOwnerFieldList())) {
                $this->getMeta('Users');
                $this->getMeta('Groups');
            }
        }

        $query = $this->getSelectClause();
        $query .= $this->getFromClause();
        $query .= $this->getWhereClause();

        $this->query = $this->updateQuery($query);

        return $this->query;
    }

    /**
     * @param string $fieldName
     * @param string $searchValue
     */
    public function addContainCondition($fieldName, $searchValue)
    {
        if ($this->conditionInstanceCount > 0) {
            $this->startGroup(QueryGenerator::$OR);
        } else {
            $this->startGroup('');
        }

        $this->addCondition($fieldName, $searchValue, 'c');
        $this->endGroup();
    }

    public function updateQuery($query)
    {
        return str_replace(array(
            'INNER JOIN vtiger_freetagged_objects',
            'INNER JOIN vtiger_freetags',
        ), array(
            'LEFT JOIN vtiger_freetagged_objects',
            'LEFT JOIN vtiger_freetags',
        ), $query);
    }

    public function isFromClauseColumnRequired($leftJoinTable)
    {
        if (false !== stripos($this->getWhereClause(), $leftJoinTable)) {
            return true;
        }

        if (false !== stripos($this->getSelectClauseColumnSQL(), $leftJoinTable)) {
            return true;
        }

        return false;
    }

    /**
     * @throws Exception
     */
    public function getFromClause()
    {
        parent::getFromClause();

        $moduleName = $this->meta->getEntityName();
        $moduleModel = Vtiger_Module_Model::getInstance($moduleName);

        foreach ($moduleModel->getFields() as $fieldModel) {
            if ($fieldModel && $fieldModel->isReferenceField()) {
                $this->setDynamicReferenceTables($fieldModel);
            }
        }

        return $this->fromClause;
    }

    /**
     * @param Vtiger_Field_Model $fieldModel
     * @throws Exception
     */
    public function setDynamicReferenceTables($fieldModel)
    {
        foreach ($fieldModel->getReferenceList() as $referenceModule) {
            $this->setReferenceTablesFromModule($fieldModel, $referenceModule);
        }
    }

    /**
     * @param Vtiger_Field_Model $fieldModel
     * @param string $referenceModule
     * @throws Exception
     */
    public function setReferenceTablesFromModule($fieldModel, $referenceModule)
    {
        $fieldColumn = $fieldModel->get('column');
        $fieldTable = $fieldModel->get('table');
        $fieldName = $fieldModel->getName();
        $referenceModuleFocus = CRMEntity::getInstance($referenceModule);

        foreach ($referenceModuleFocus->tab_name_index as $table => $tableId) {
            $tableAlias = sprintf('%s%s', $table, $fieldName);

            if ($this->isJoinedTable($table, $tableAlias) && $this->isFromClauseColumnRequired($tableAlias)) {
                $this->fromClause .= sprintf(' LEFT JOIN %s AS %s ON %s.%s = %s.%s ',
                    $table, $tableAlias, $tableAlias, $tableId, $fieldTable, $fieldColumn
                );

                if ('vtiger_crmentity' === $table) {
                    $this->fromClause .= sprintf('AND %s.deleted=0 ', $tableAlias);
                }
            }
        }
    }

    public function isJoinedTable($table, $tableAlias)
    {
        $pattern = sprintf('/(left|LEFT|inner|INNER)\s*(join|JOIN)\s*(%s)\s*(as|AS)*\s*(%s)/', addslashes($table), addslashes($tableAlias));

        preg_match_all($pattern, $this->fromClause, $matches, PREG_SET_ORDER, 0);

        return empty($matches);
    }

    public function getSQLSelectColumn($name)
    {
        if ('id' === $name) {
            $baseTable = $this->meta->getEntityBaseTable();
            $moduleTableIndexList = $this->getTableIndexList();
            $baseTableIndex = $moduleTableIndexList[$baseTable];

            return $baseTable . '.' . $baseTableIndex;
        }

        $moduleFields = $this->getModuleFields();
        $field = $moduleFields[$name];

        $referenceField = '';
        preg_match('/(\w+) ; \((\w+)\) (\w+)/', $name, $matches);

        if (0 !== count($matches)) {
            list($full, $referenceField, $referenceModule, $fieldname) = $matches;
        }

        return $field->getTableName() . $referenceField . '.' . $field->getColumnName();
    }


    public function getSelectClause()
    {
        if (!empty($this->selectClause)) {
            return $this->selectClause;
        }

        $columns = [
            'vtiger_crmentity.crmid',
            'vtiger_crmentity.crmid AS vtiger_crmentity_crmid',
            'vtiger_crmentity.setype',
            'vtiger_crmentity.setype AS vtiger_crmentity_setype',
            'vtiger_crmentity.label',
            'vtiger_crmentity.createdtime',
        ];
        $moduleFields = $this->getModuleFields();
        $accessibleFieldList = array_keys($moduleFields);
        $accessibleFieldList[] = 'id';

        $this->fields = array_intersect($accessibleFieldList, $this->fields);

        foreach ($this->fields as $field) {
            $sql = $this->getSQLSelectColumn($field);

            //To merge date and time fields
            if ('Calendar' === $this->meta->getEntityName() && (in_array($field, ['date_start', 'due_date', 'taskstatus', 'eventstatus',]))) {
                if ('date_start' === $field) {
                    $timeField = 'time_start';
                    $sql = $this->getSQLColumn($timeField);
                } elseif ('due_date' === $field) {
                    $timeField = 'time_end';
                    $sql = $this->getSQLColumn($timeField);
                } elseif (in_array($field, ['taskstatus', 'eventstatus',])) {
                    //In calendar list view, Status value = Planned is not displaying
                    $sql = "CASE WHEN (vtiger_activity.status not like '') THEN vtiger_activity.status ELSE vtiger_activity.eventstatus END AS ";
                    if ('taskstatus' === $field) {
                        $sql .= "status";
                    } else {
                        $sql .= $field;
                    }
                }

                $columns[] = $sql;
            }
        }

        $this->columns = $columns;
        $this->selectClause = sprintf('SELECT %s ' , implode(', ', $columns));

        return $this->selectClause;
    }

    public function getWhereClause()
    {
        global $current_user;
        if (!empty($this->query) || !empty($this->whereClause)) {
            return $this->whereClause;
        }
        $deletedQuery = $this->meta->getEntityDeletedQuery();
        $sql = '';
        if (!empty($deletedQuery)) {
            $sql .= " WHERE $deletedQuery";
        }
        if ($this->conditionInstanceCount > 0) {
            $sql .= ' AND (';
        } elseif (empty($deletedQuery)) {
            $sql .= ' WHERE ';
        }
        $baseModule = $this->getModule();
        $moduleFieldList = $this->getModuleFields();
        $baseTable = $this->meta->getEntityBaseTable();
        $moduleTableIndexList = $this->meta->getEntityTableIndexList();
        $baseTableIndex = $moduleTableIndexList[$baseTable];
        $groupSql = $this->groupInfo;

        $fieldSqlList = array();
        foreach ($this->conditionals as $index => $conditionInfo) {
            $fieldName = $conditionInfo['name'];
            
            /* @var $field WebserviceField */
            $field = $moduleFieldList[$fieldName];

            if (empty($field) || $conditionInfo['operator'] == 'None') {
                continue;
            }

            $fieldSql = '(';
            $fieldGlue = '';
            $valueSqlList = $this->getConditionValue($conditionInfo['value'], $conditionInfo['operator'], $field);
            $operator = strtolower($conditionInfo['operator']);

            if ($operator == 'between' && $this->isDateType($field->getFieldDataType())) {
                $start = explode(' ', $conditionInfo['value'][0]);
                if (count($start) == 2) {
                    $conditionInfo['value'][0] = getValidDBInsertDateTimeValue($start[0] . ' ' . $start[1]);
                }

                $end = explode(' ', $conditionInfo['values'][1]);
                // Dates will be equal for Today, Tomorrow, Yesterday.
                if (count($end) == 2) {
                    if ($start[0] == $end[0]) {
                        $dateTime = new DateTime($conditionInfo['value'][0]);
                        $nextDay = $dateTime->modify('+1 days');
                        $nextDay = $nextDay->format('Y-m-d H:i:s');
                        $values = explode(' ', $nextDay);
                        $conditionInfo['value'][1] = getValidDBInsertDateTimeValue($values[0]) . ' ' . $values[1];
                    } else {
                        $end = $conditionInfo['value'][1];
                        $dateObject = new DateTimeField($end);
                        $conditionInfo['value'][1] = $dateObject->getDBInsertDateTimeValue();
                    }
                }
            }
            if (!is_array($valueSqlList)) {
                $valueSqlList = array($valueSqlList);
            }

            $valueSqlListNums = count($valueSqlList);
            $valueSqlListNum = 0;

            foreach ($valueSqlList as $valueSql) {
                $valueSqlListNum++;

                if (in_array($fieldName, $this->getReferenceFieldList())) {
                    $refFieldInfoList = $this->getReferenceFieldInfoList();
                    $moduleList = $refFieldInfoList[$fieldName];
                    foreach ($moduleList as $module) {
                        $nameFields = $this->getMeta($module)->getNameFields();
                        $nameFieldList = explode(',', $nameFields);
                        $meta = $this->getMeta($module);
                        $columnList = array();
                        foreach ($nameFieldList as $column) {
                            if ($module == 'Users') {
                                $instance = CRMEntity::getInstance($module);
                                $referenceTable = $instance->table_name;
                                if (count($this->ownerFields) > 0 ||
                                    $this->getModule() == 'Quotes') {
                                    $referenceTable .= $fieldName;
                                }
                            } else {
                                $referenceField = $meta->getFieldByColumnName($column);
                                $referenceTable = $referenceField->getTableName() . $fieldName;
                            }
                            if (isset($moduleTableIndexList[$referenceTable])) {

                                $referenceTable = "$referenceTable$fieldName";
                            }
                            $columnList[$column] = "$referenceTable.$column";
                        }

                        if (count($columnList) > 1) {
                            $columnSql = getSqlForNameInDisplayFormat($columnList, $module);
                        } else {
                            $columnSql = implode('', $columnList);
                        }

                        $fieldSql .= sprintf('%s trim(%s) %s ', $fieldGlue, $columnSql, $valueSql);
                        $fieldGlue = ' OR';
                    }
                }

                $fieldGlueAfter = ($valueSqlListNum < $valueSqlListNums) ? ' OR ' : '';
                $fieldSql .= sprintf('%s %s.%s %s %s', $fieldGlue, $this->getTableName($field), $this->getColumnName($field), $valueSql, $fieldGlueAfter);
            }

            $fieldSql .= ')';
            $fieldSqlList[$index] = $fieldSql;
        }

        ksort($fieldSqlList);
        $groupSql = $this->makeGroupSqlReplacements($fieldSqlList, $groupSql);

        if ($this->conditionInstanceCount > 0) {
            $groupSql .= " OR vtiger_crmentity.label LIKE '%" . $this->getSearchValue() . "%' ";
            $this->conditionalWhere = $groupSql;
            $sql .= $groupSql . ')';
        } else {
            $sql .= " AND vtiger_crmentity.label LIKE '%" . $this->getSearchValue() . "%' ";
        }

        $sql .= " AND $baseTable.$baseTableIndex > 0 AND vtiger_crmentity.setype = '$baseModule' ";

        $this->whereClause = $sql;

        return $sql;
    }

    /**
     * @param WebserviceField $field
     * @return string
     */
    public function getTableName($field)
    {
        if('tags' === $field->getFieldName()) {
            return 'vtiger_freetags';
        }

        return $field->getTableName();
    }

    /**
     * @param WebserviceField $field
     * @return string
     */
    public function getColumnName($field)
    {
        if('tags' === $field->getFieldName()) {
            return 'tag';
        }

        return $field->getColumnName();
    }

    protected function getConditionValue($value, $operator, $field)
    {
        $operator = strtolower($operator);
        $db = PearDatabase::getInstance();
        $fieldDataType = $field->getFieldDataType();

        if (is_string($value) && $this->ignoreComma == false) {
            $commaSeparatedFieldTypes = array('picklist', 'multipicklist', 'owner', 'date', 'datetime', 'time');

            if (in_array($fieldDataType, $commaSeparatedFieldTypes)) {
                $valueArray = explode(',', $value);

                if ($fieldDataType == 'multipicklist' && in_array($operator, array('e', 'n'))) {
                    $valueArray = getCombinations($valueArray);

                    foreach ($valueArray as $key => $value) {
                        $valueArray[$key] = ltrim($value, ' |##| ');
                    }
                }
            } else {
                $valueArray = array($value);
            }
        } elseif (is_array($value)) {
            $valueArray = $value;
        } else {
            $valueArray = array($value);
        }

        $sql = array();

        if ($operator == 'between' || $operator == 'bw' || $operator == 'notequal') {
            if ($field->getFieldName() == 'birthday') {
                $valueArray[0] = getValidDBInsertDateTimeValue($valueArray[0]);
                $valueArray[1] = getValidDBInsertDateTimeValue($valueArray[1]);
                $sql[] = "BETWEEN DATE_FORMAT(" . $db->quote($valueArray[0]) . ", '%m%d') AND " .
                    "DATE_FORMAT(" . $db->quote($valueArray[1]) . ", '%m%d')";
            } else {
                if ($this->isDateType($fieldDataType)) {
                    $start = explode(' ', $valueArray[0]);
                    $end = explode(' ', $valueArray[1]);
                    if ($operator == 'between' && count($start) == 2 && count($end) == 2) {
                        $valueArray[0] = getValidDBInsertDateTimeValue($start[0] . ' ' . $start[1]);

                        if ($start[0] == $end[0]) {
                            $dateTime = new DateTime($valueArray[0]);
                            $nextDay = $dateTime->modify('+1 days');
                            $nextDay = strtotime($nextDay->format('Y-m-d H:i:s')) - 1;
                            $nextDay = date('Y-m-d H:i:s', $nextDay);
                            $values = explode(' ', $nextDay);
                            $valueArray[1] = getValidDBInsertDateTimeValue($values[0]) . ' ' . $values[1];
                        } else {
                            $end = $valueArray[1];
                            $dateObject = new DateTimeField($end);
                            $valueArray[1] = $dateObject->getDBInsertDateTimeValue();
                        }
                    } else {
                        $valueArray[0] = getValidDBInsertDateTimeValue($valueArray[0]);
                        $dateTimeStart = explode(' ', $valueArray[0]);
                        if ($dateTimeStart[1] == '00:00:00' && $operator != 'between') {
                            $valueArray[0] = $dateTimeStart[0];
                        }
                        $valueArray[1] = getValidDBInsertDateTimeValue($valueArray[1]);
                        $dateTimeEnd = explode(' ', $valueArray[1]);
                        if ($dateTimeEnd[1] == '00:00:00' || $dateTimeEnd[1] == '23:59:59') {
                            $valueArray[1] = $dateTimeEnd[0];
                        }
                    }
                }

                if ($operator == 'notequal') {
                    $sql[] = "NOT BETWEEN " . $db->quote($valueArray[0]) . " AND " .
                        $db->quote($valueArray[1]);
                } else {
                    $sql[] = "BETWEEN " . $db->quote($valueArray[0]) . " AND " .
                        $db->quote($valueArray[1]);
                }
            }
            return $sql;
        }
        foreach ($valueArray as $value) {
            if (!$this->isStringType($fieldDataType)) {
                $value = trim($value);
            }
            if ($operator == 'empty' || $operator == 'y') {
                $sql[] = sprintf("IS NULL OR %s = ''", $this->getSQLColumn($field->getFieldName()));
                continue;
            }
            if ($operator == 'ny') {
                $sql[] = sprintf("IS NOT NULL AND %s != ''", $this->getSQLColumn($field->getFieldName()));
                continue;
            }
            if ((strtolower(trim($value)) == 'null') ||
                (trim($value) == '' && !$this->isStringType($fieldDataType)) &&
                ($operator == 'e' || $operator == 'n')) {
                if ($operator == 'e') {
                    $sql[] = "IS NULL";
                    continue;
                }
                $sql[] = "IS NOT NULL";
                continue;
            } elseif ($fieldDataType == 'boolean') {
                $value = strtolower($value);
                if ($value == 'yes') {
                    $value = 1;
                } elseif ($value == 'no') {
                    $value = 0;
                }
            } elseif ($this->isDateType($fieldDataType)) {
                // For "after" and "before" conditions
                $values = explode(' ', $value);
                if (($operator == 'a' || $operator == 'b') && count($values) == 2) {
                    if ($operator == 'a') {
                        // for after comparator we should check the date after the given
                        $dateTime = new DateTime($value);
                        $modifiedDate = $dateTime->modify('+1 days');
                        $nextday = $modifiedDate->format('Y-m-d H:i:s');
                        $temp = strtotime($nextday) - 1;
                        $date = date('Y-m-d H:i:s', $temp);
                        $value = getValidDBInsertDateTimeValue($date);
                    } else {
                        $dateTime = new DateTime($value);
                        $prevday = $dateTime->format('Y-m-d H:i:s');
                        $temp = strtotime($prevday) - 1;
                        $date = date('Y-m-d H:i:s', $temp);
                        $value = getValidDBInsertDateTimeValue($date);
                    }
                } else {
                    $value = getValidDBInsertDateTimeValue($value);
                    $dateTime = explode(' ', $value);
                    if ($dateTime[1] == '00:00:00') {
                        $value = $dateTime[0];
                    }
                }
            } elseif ('currency' === $fieldDataType) {
                $uiType = (int)$field->getUIType();

                if (72 === $uiType) {
                    $value = CurrencyField::convertToDBFormat($value, null, true);
                } elseif (71 === $uiType) {
                    $value = CurrencyField::convertToDBFormat($value);
                }
            }

            $value = $db->sql_escape_string($value);

            if(empty(trim($value))) {
                if (in_array($operator, ['s', 'ew', 'c']) && ($this->isStringType($fieldDataType) || $this->isPicklistType($fieldDataType))) {
                    $sql[] = "LIKE ''";
                    continue;
                }

                if (($operator == 'k') && $this->isStringType($fieldDataType)) {
                    $sql[] = "NOT LIKE ''";
                    continue;
                }

                if ($this->isNumericType($fieldDataType)) {
                    $value = '0';
                }
            }

            switch ($operator) {
                case 'e':
                    $sqlOperator = '=';
                    break;
                case 'n':
                    $sqlOperator = '<>';
                    break;
                case 's':
                    $sqlOperator = 'LIKE';
                    $value = "$value%";
                    break;
                case 'ew':
                    $sqlOperator = 'LIKE';
                    $value = "%$value";
                    break;
                case 'c':
                    $sqlOperator = 'LIKE';
                    $value = "%$value%";
                    break;
                case 'k':
                    $sqlOperator = 'NOT LIKE';
                    $value = "%$value%";
                    break;
                case 'b':
                case 'l':
                    $sqlOperator = '<';
                    break;
                case 'a':
                case 'g':
                    $sqlOperator = '>';
                    break;
                case 'm':
                    $sqlOperator = '<=';
                    break;
                case 'h':
                    $sqlOperator = '>=';
                    break;
            }

            $sql[] = "$sqlOperator '$value'";
        }

        return $sql;
    }

    public function isPicklistType($type) {
        return in_array($type, ['picklist', 'multipicklist']);
    }

    /**
     * @param string $type
     * @return bool
     */
    public function isDateType($type)
    {
        return in_array($type, ['date', 'datetime']);
    }

    /**
     * @param object $fieldModel
     * @param string $searchValue
     * @return bool
     */
    public function isValidSearchValueForField($fieldModel, $searchValue)
    {
        $fieldType = $fieldModel->getFieldDataType();

        if ($this->isDateType($fieldType) && !$this->isValidDate($searchValue)) {
            return false;
        }

        if($this->isNumericType($fieldType) && !$this->isValidNumeric($searchValue)) {
            return false;
        }

        if($this->isTimeType($fieldType) && !$this->isValidTime($searchValue)) {
            return false;
        }

        return true;
    }

    public function isTimeType($type)
    {
        return 'time' === $type;
    }

    public function isValidTime($value)
    {
        preg_match_all(
            '/^[\d]{2}([:][\d]{2})*([:][\d]{2})*$/',
            html_entity_decode($value),
            $matches,
            PREG_SET_ORDER
            , 0
        );

        return !empty($matches);
    }

    public function isValidNumeric($value)
    {
        preg_match_all(
            '/^[0-9]*(?:[^0-9a-z][0-9]*)*$/',
            html_entity_decode($value),
            $matches,
            PREG_SET_ORDER
            , 0
        );

        return !empty($matches);
    }

    public function isValidDate($value)
    {
        preg_match_all(
            '/^[\d]{1,4}[^\d][\d]{1,4}[^\d][\d]{1,4}[ ]*([\d]{1,2}[^\d][\d]{1,2})*([^\d][\d]{1,2})*$/',
            html_entity_decode($value),
            $matches,
            PREG_SET_ORDER
            ,0
        );

        return !empty($matches);
    }

    protected function isStringType($type)
    {
        return ($type == 'string' || $type == 'text' || $type == 'email' || $type == 'reference');
    }

    protected function isRelativeSearchOperators($operator)
    {
        $nonDaySearchOperators = array('l', 'g', 'm', 'h');
        return in_array($operator, $nonDaySearchOperators);
    }

    protected function isNumericType($type)
    {
        return ($type == 'integer' || $type == 'double' || $type == 'currency');
    }

    protected function makeGroupSqlReplacements($fieldSqlList, $groupSql)
    {
        $pos = 0;
        $nextOffset = 0;
        foreach ($fieldSqlList as $index => $fieldSql) {
            $pos = strpos($groupSql, $index . '', $nextOffset);

            if ($pos !== false) {
                $beforeStr = substr($groupSql, 0, $pos);
                $afterStr = substr($groupSql, $pos + strlen($index));
                $nextOffset = strlen($beforeStr . $fieldSql);
                $groupSql = $beforeStr . $fieldSql . $afterStr;
            }
        }

        return $groupSql;
    }

    /**
     * @param array $fieldTableAndColumns
     */
    public function initForCustomSearchColumns($fieldTableAndColumns)
    {
        $fields = array();

        foreach ($fieldTableAndColumns as $fieldTableAndColumn) {
            list($table, $column) = explode('::', $fieldTableAndColumn);

            $fields[] = $column;
        }

        parent::setFields($fields);
    }

    /**
     * @param array $fieldModels
     */
    public function initForCustomSearchFields($fieldModels) {
        $fields = array();

        foreach ($fieldModels as $fieldModel) {
            $fields[] = $fieldModel->get('name');
        }

        parent::setFields($fields);
    }

    public static $columnFieldNames = array();

    /**
     * @param string $column
     * @param string $table
     * @return string
     * @throws Exception
     */
    public static function getFieldNameFromColumn($table, $column)
    {
        if (!isset(self::$columnFieldNames[$table][$column])) {
            $adb = PearDatabase::getInstance();
            $result = $adb->pquery('SELECT fieldname FROM vtiger_field WHERE columnname=? AND tablename=?',
                [$column, $table]
            );
            self::$columnFieldNames[$table][$column] = $adb->query_result($result, 0, 'fieldname');
        }

        return self::$columnFieldNames[$table][$column];
    }

    /**
     * @param string $value
     * @return string
     * @throws Exception
     */
    public static function getFieldNameFromString($value)
    {
        $fieldData = explode('::', $value);

        return self::getFieldNameFromColumn($fieldData[0], $fieldData[1]);
    }

    public function setSearchValue($value)
    {
        $this->searchValue = $value;
    }

    public function getSearchValue()
    {
        return $this->searchValue;
    }
}
