<?php
/*+***********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.1
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * Portions created by Libertus Solutions are Copyright (C) Libertus Solutions
 * All Rights Reserved.
 *************************************************************************************/
 
/**
 * This View class is the core of GeoTools which manages the user
 * interaction of the embeded ListView inside the GeoTools page display
 */
 
class GeoTools_GeoTools_View extends Vtiger_Index_View {

    var $defaultModule = false;
    var $defaultUnit = false;
    var $defaultRadius = false;
    var $defaultZoom = false;
    var $defaultMapCenter = false;
    var $defaultConstraint = 'Inside';
    var $defaultGeoSortOrder = 'Nearest';
    
    var $mapCenter = false;
    var $companyCenter = false;
    var $mapProvider = false;
    var $mapJSClass = false;
    var $iconPath = false;
    var $distanceUnits = false;
    
    var $fallback = false;
    
    var $configuredModulesData = array();
    var $centerOptions = array();
    var $settings = array();

    var $pagingModel = false;

    function __construct() {
        // Set up all the default params
        $searchModel = GeoTools_Search_Model::getInstance();
        $this->defaultUnit = $searchModel->defaults['distance'];
        $this->defaultRadius = $searchModel->defaults['radius'];
        $this->defaultModule = $searchModel->defaults['defaultmod'];
        $this->defaultZoom = $searchModel->defaults['zoom'];
        $this->defaultMapCenter = $searchModel->defaults['defaultcentre'];
        $this->mapCenter = $searchModel->defaults['mapcentre'];
        $this->companyCenter = $searchModel->defaults['companyloc'];
        $this->userCenter = $searchModel->getUserAddressLocation();
        $this->iconPath = $searchModel->defaults['iconpath'];
        $this->listEntries = $searchModel->defaults['listentries'];
        
        $this->distanceUnits = $searchModel->units;
        $this->geoCoder = $searchModel->defaults['geocoder'];
        $this->mapProvider = $searchModel->defaults['mapservice'];
        $this->mapJSClass = $searchModel->defaults['mapJS_URL'];
        $this->configuredModulesData = $searchModel->getConfiguredModules();
        $this->centerOptions = $searchModel->defaultcentre;
        $this->fallback = $searchModel->fallback;
        
        // If user's address hasn't been decoded remove option from Picklist
        if(!$this->userCenter) {
            unset($this->centerOptions['3']);
        }
   }

   public function preProcess (Vtiger_Request $request, $display=true) {
       $vtigerCurrentVersion = vglobal('vtiger_current_version');
        if($vtigerCurrentVersion < 7) {
            parent::preProcess($request, true);
        } else {
            parent::preProcess($request, false);
            $viewer = $this->getViewer($request);

            // The module for the embedded ListView
            if ($request->get('geotoolsModule')) {
                $listModuleName = $request->get('geotoolsModule');
            } else {
                $listModuleName = $this->defaultModule;
            }
            if(!$listModuleName) {
                $viewer->assign('MESSAGE', vtranslate('LBL GEOTOOLS NO MODULES CONFIGURED', 'GeoTools'));
                $viewer->view('OperationNotPermitted.tpl', 'Vtiger');
                exit;
            }

            // Get the embedded Module's filter view id
            // Fix for Events/Calendar module!
            $tempModuleName = $listModuleName;
            if($listModuleName == 'Events') {
                $listModuleName = 'Calendar';
            }
            $configuredModulesData = $this->configuredModulesData;
            $this->listViewName = $request->get('viewname');
            if(empty($this->listViewName)) {
                if($configuredModulesData[$listModuleName]['default_filter']['fieldid']) {
                    $this->listViewName = $configuredModulesData[$listModuleName]['default_filter']['fieldid'];
                } else {
                    $listCustomView = new CustomView();
                    if($listModuleName == 'Events') {
                        $this->listViewName = $listCustomView->getViewId('Calendar');
                    } else {
                        $this->listViewName = $listCustomView->getViewId($listModuleName);
                    }
                }
            }
            $customView = new CustomView();

            if($request->get('viewname') == 'noGeoCodes') {
                // Don't test for permissions for our virtual Custom View
            } elseif($customView->isPermittedCustomView($request->get('viewname'), 'List', $listModuleName) != 'yes') {
                $viewName = $customView->getViewIdByName('All', $listModuleName);
                $request->set('viewname', $viewName);
                $_REQUEST['viewname'] = $viewName;
            }

            $cvId = $this->viewName;
            if(!$cvId) {
                $cvId = $customView->getViewId($listModuleName);
            }

            $listHeaders = $request->get('list_headers', array());
            $tag = $request->get('tag');
            $listViewSessionKey = $listModuleName.'_'.$cvId;

            if(!empty($tag)) {
                $listViewSessionKey .='_'.$tag;
            }

            $orderParams = Vtiger_ListView_Model::getSortParamsSession($listViewSessionKey);

            if(empty($listHeaders)) {
                $listHeaders = $orderParams['list_headers'];
            }

            $this->listViewModel = Vtiger_ListView_Model::getInstance($listModuleName, $cvId, $listHeaders);
            $linkParams = array('MODULE'=>$listModuleName, 'ACTION'=>$request->get('view'));
            $viewer->assign('CUSTOM_VIEWS', CustomView_Record_Model::getAllByGroup($listModuleName));
            $this->viewName = $request->get('viewname');
            if(empty($this->viewName)){
                //If not view name exits then get it from custom view
                //This can return default view id or view id present in session
                $customView = new CustomView();
                $this->viewName = $customView->getViewId($listModuleName);
            }

            // Reset ListViewModule name for Events/Calendar module 
            $listModuleName = $tempModuleName;

            $quickLinkModels = $this->listViewModel->getSideBarLinks($linkParams);
            $viewer->assign('QUICK_LINKS', $quickLinkModels);
            $viewer->assign('VIEWID', $this->viewName);
            $listModuleModel = Vtiger_Module_Model::getInstance($listModuleName);
            $viewer->assign('LIST_MODULE_MODEL', $listModuleModel);
            $geoToolsModuleModel = Vtiger_Module_Model::getInstance('GeoTools');
            $viewer->assign('MODULE_MODEL', $geoToolsModuleModel);
            $viewer->assign('VIEWNAME', $this->listViewName);
            $viewer->assign('LIST_VIEWID', $this->listViewName);
            if($display) {
                $this->preProcessDisplay($request);
            }
        }
    }

    /**
     * Function to returns the preProcess Template Name
     * @param <type> $request
     * @return <String>
     */
    public function preProcessTplName(Vtiger_Request $request) {    
        return 'GeoToolsPreProcessView.tpl';
    }
    
    public function process(Vtiger_Request $request) {
		
        $moduleName = $request->getModule();
        $viewer = $this->getViewer($request);
        
        $viewer->assign('QUALIFIED_MODULE', $moduleName);
        $this->listProcess($request);
        if($request->has('_pjax')){
            $viewer->view('GeoToolsListViewContents.tpl', $moduleName);
        } else {
            $viewer->view('Index.tpl', $moduleName);
        }
    }
    
    public function listProcess(Vtiger_Request $request) {

        $moduleName = $request->getModule();
        
        // Radius search from another entity record
        if($request->get('src_module') && $request->get('src_record')) {
            $searchModel = GeoTools_Search_Model::getInstance();

            $module = $request->get('src_module');
            $crmid = $request->get('src_record');
            $location = $searchModel->getLatLng($module, $crmid);
            
            if ($location) {
                // Set the Map Centre & Picklist option to Entity record
                $request->set('geotoolsMapCenter', $location);
                $request->set('geotoolsCenterOption', '5');
            }
            
            // We have a target module as part of the action link
            if ($request->get('tgt_module')) {
                $request->set('geotoolsModule', $request->get('tgt_module'));
            }
        }
        
        // Check the Request params and override the defaults if set
        
        // Set up the Picklist options on the GeoTools Search
        $configuredModulesData = $this->configuredModulesData;
        $configuredModules = array_keys($configuredModulesData);
        
        // GeoCoding Provider
    	$this->viewer->assign('GEOCODER_PROVIDER', $this->geoCoder);
        
        // Might want to have the option of a separate Mapping Provider in the future too
        $this->viewer->assign('MAP_PROVIDER', $this->mapProvider);
        
        // Link to mapping provider's javascript library
        $this->viewer->assign('MAPJS_URL', $this->mapJSClass);
        
        $this->viewer->assign('CONFIGURED_MODULES', $configuredModules);
        $this->viewer->assign('DISTANCE_UNITS',  $this->distanceUnits);
        $this->viewer->assign('CENTER_OPTIONS', $this->centerOptions);
                
        // Set the radius centre icon
        $this->viewer->assign('CENTERICON', $this->iconPath);
        
        // Set the default map zoom
        $this->viewer->assign('ZOOM', $this->defaultZoom);
        
        // The module for the embedded ListView
        if($request->get('geotoolsModule')){
            $listModuleName = $request->get('geotoolsModule');
        } else {
            $listModuleName = $this->defaultModule;
        }
        
        $this->viewer->assign('ACTIVE_MODULE', $listModuleName);
        
        $this->viewer->assign('ICONPATH', $configuredModulesData[$listModuleName]['iconpath']['fieldid']);
        
        // The Map Centre Option 
        // (If this changes we need to remove the $request['geotoolsMapCenter'] as it will be recast.
        if($request->get('geotoolsCenterOption')) {
            $centerOption = $request->get('geotoolsCenterOption');
        } else {
            $centerOption = $this->defaultMapCenter;
        }
        $this->viewer->assign('ACTIVE_CENTER_OPTION', $centerOption);
        
        // Centre of Map - affected by the Map Centre Select option
        if($request->get('geotoolsMapCenter')) {
            $center = $request->get('geotoolsMapCenter');
        } else {
            // Probably need a case select here
            if($centerOption == '1') {
                $center = $this->mapCenter;
            } else if($centerOption == '2') {
                $center = $this->companyCenter;
            } else if($centerOption == '3') {
                $center = $this->userCenter;
            }
        }

        if (!$center) {
            $center = $this->fallback;      
        }

        $this->settings['mapcentre'] = $center; // This is the lat/lng that gets passed to the query generator!
        $this->viewer->assign('ACTIVE_MAPCENTER', $center);
        
        // If no List module drop back here
        if(!$listModuleName) {
            return;
        }
        
        // Distance Units
        if($request->get('geotoolsDistanceUnits')){
            $unit = $request->get('geotoolsDistanceUnits');
        } else {
            $unit = $this->defaultUnit;
        }
        $this->settings['distance'] = $unit;
        $this->viewer->assign('ACTIVE_UNIT', $unit);
        
        // Search Radius
        if($request->get('geotoolsRadius')){
            $radius = $request->get('geotoolsRadius');
        } else {
            $radius = $this->defaultRadius;
        }
        $this->settings['radius'] = $radius;
        $this->viewer->assign('ACTIVE_RADIUS', $radius);

        // Radius Constraint (Inside or outside the Radius)
        if($request->get('geotoolsConstraint')){
            $constraint = $request->get('geotoolsConstraint');
        } else {
            $constraint = $this->defaultConstraint;
        }
        $this->settings['constraint'] = $constraint;
        $this->viewer->assign('ACTIVE_CONSTRAINT', $constraint);
        
        // geotools Sort Order (Nearest or Farthest first)
        if($request->get('geotoolsSortOrder')){
            $geotoolsSortOrder = $request->get('geotoolsSortOrder');
        } else {
            $geotoolsSortOrder = $this->defaultGeoSortOrder;
        }
        $this->settings['geotoolsSortOrder'] = $geotoolsSortOrder;
        $this->viewer->assign('ACTIVE_GEOSORTORDER', $geotoolsSortOrder);
        
        // Order By
        if($request->get('orderby')){
            $orderby = $request->get('orderby');
        } else {
            $orderby = '';
        }
        $this->settings['orderby'] = $orderby;
        $this->viewer->assign('ACTIVE_ORDERBY', $orderby);
        
        // Sort Order
        if($request->get('sortorder')){
            $sortorder = $request->get('sortorder');
        } else {
            $sortorder = '';
        }
        $this->settings['sortorder'] = $sortorder;
        $this->viewer->assign('ACTIVE_SORTORDER', $sortorder);
        
        // Create a new request object for the embedded list
        $listRequestArray = array();
        $values = array('embedModule', 'module', 'operator', 'orderby', 'page', 
                        'parent', 'search_key', 'search_value', 'sortorder', 
                        'search_params', 'view', 'viewname');
        foreach($values as $value){
            if($request->has($value)){
                $listRequestArray[$value] = $request->get($value);
            }
        }       
        $listRequestArray['module'] = $listModuleName;
        $listRequestArray['view'] = 'List';
        $listRequest = new Vtiger_Request($listRequestArray, $listRequestArray);

        // Create the embeddedList Viewer object
        $listViewer = $this->getViewer($listRequest);
        
        // The embedded Module's model Object
        $listModuleModel = Vtiger_Module_Model::getInstance($listModuleName);
        
        // Setup picklist field name to Query Generator
        if($this->configuredModulesData[$listModuleName]['icon_picklist']['fieldid']) {
            $this->settings['icon_picklist'] = $this->configuredModulesData[$listModuleName]['icon_picklist']['fieldid'];
            $fieldModel = Vtiger_Field_Model::getInstance($this->settings['icon_picklist'], $listModuleModel);
            //print_r($fieldModel);
            // Add to template vars
            $this->viewer->assign('ICON_PICKLIST', vtranslate($fieldModel->get('label'), $listModuleName) );
        }

        // Get the embedded Module's filter view id
        $this->listViewName = $listRequest->get('viewname');
        if(empty($this->listViewName)) {
            if($configuredModulesData[$listModuleName]['default_filter']['fieldid']) {
                $this->listViewName = $configuredModulesData[$listModuleName]['default_filter']['fieldid'];
            } else {
                $listCustomView = new CustomView();
                $this->listViewName = $listCustomView->getViewId($listModuleName);
            }
        }
        // Set a place Holder for the viewname - 
        // if noGeoCodes change the zero result page message
        $this->viewer->assign('VIEWNAME', $this->listViewName);
        $this->viewer->assign('LIST_VIEWID', $this->listViewName);
        
        $this->initializeListViewContents($listRequest, $listViewer);
		$this->assignCustomViews($listRequest, $listViewer);

		
        $listViewer->assign('VIEW', $listRequest->get('view'));
        $listViewer->assign('LIST_MODULE_MODEL', $listModuleModel);
        $listInstance = new Vtiger_List_View();
        if(function_exists('getRecordActionsFromModule')){
            $listViewer->assign('RECORD_ACTIONS', $listInstance->getRecordActionsFromModule($listModuleModel));
        }
        if(function_exists('isQuickPreviewEnabled')){
            if($listModuleModel->isQuickPreviewEnabled()) {
                $listViewer->assign('QUICK_PREVIEW_ENABLED', 'true');
            }
        }
        $listViewer->assign('CURRENT_USER_MODEL', Users_Record_Model::getCurrentUserModel());
    }

    // Injecting custom javascript resources
    public function getHeaderScripts(Vtiger_Request $request) {

        // Get the correct List View Class for the embedded List module so we 
        // get the right Javascript classes.
        $geotoolsModule = $request->get('geotoolsModule');
        if(empty($geotoolsModule)){
            $geotoolsModule = 'GeoTools';
        }
        $listRequestArray['module'] = $geotoolsModule;
        $listRequestArray['view'] = 'List';
        $listRequest = new Vtiger_Request($listRequestArray, $listRequestArray);
        $listModuleListView = new Vtiger_List_View();
        $headerScriptInstances = $listModuleListView->getHeaderScripts($listRequest);
        $moduleJSFile = "modules.$geotoolsModule.resources.$geotoolsModule";
        if($geotoolsModule == 'Events') {
            $moduleJSFile = "modules.$geotoolsModule.resources.Edit";
        }
        unset($headerScriptInstances[$moduleJSFile]);
        // Get correct Map Provider class
        $providerClass = 'modules.GeoTools.resources.' . $this->mapProvider . 'mapCode';
        $jsFileNames = array(
            'modules.Vtiger.resources.List',
            'modules.GeoTools.resources.List',
            'modules.GeoTools.resources.GeoTools',
            'modules.GeoTools.resources.Popup',
            $providerClass,
            'modules.CustomView.resources.CustomView',  
			//for vtiger7 
			"modules.Vtiger.resources.MergeRecords",
			"~layouts/".Vtiger_Viewer::getLayoutName()."/lib/jquery/Lightweight-jQuery-In-page-Filtering-Plugin-instaFilta/instafilta.min.js",
			'modules.Vtiger.resources.Tag',
			"~layouts/".Vtiger_Viewer::getLayoutName()."/lib/jquery/floatThead/jquery.floatThead.js",
			"~layouts/".Vtiger_Viewer::getLayoutName()."/lib/jquery/perfect-scrollbar/js/perfect-scrollbar.jquery.js",
        );
        
        if($this->mapProvider == 'OSM') {
            $jsFileNames[] = 'modules.GeoTools.resources.leaflet';
            $jsFileNames[] = 'modules.GeoTools.resources.leaflet-fullscreen';
            $jsFileNames[] = 'modules.GeoTools.resources.leaflet-vector-markers';
        }
        
        $jsScriptInstances = $this->checkAndConvertJsScripts($jsFileNames);
        $headerScriptInstances = array_merge($headerScriptInstances, $jsScriptInstances);
        return $headerScriptInstances;
    }
    
    public function getHeaderCss(Vtiger_Request $request) {
        // Get the List View Class for the embedded List module so we 
        // load the correct CSS files.
        $geotoolsModule = $request->get('geotoolsModule');
        if(empty($geotoolsModule)) {
            $geotoolsModule = 'GeoTools';
        }
        $listRequestArray['module'] = $geotoolsModule;
        $listRequestArray['view'] = 'List';
        $listRequest = new Vtiger_Request($listRequestArray, $listRequestArray);
        $listModuleListView = new Vtiger_List_View();
        $headerCssInstances = $listModuleListView->getHeaderCss($listRequest);
        $defaultLayoutName = Vtiger_Viewer::getLayoutName();
    
        $cssFileNames = array(
            "~/layouts/$defaultLayoutName/modules/GeoTools/resources/GeoTools.css",
        );
        
        if($this->mapProvider == 'OSM') {
            $cssFileNames[] = "~/layouts/$defaultLayoutName/modules/GeoTools/resources/leaflet.css";
            $cssFileNames[] = "~/layouts/$defaultLayoutName/modules/GeoTools/resources/leaflet-fullscreen.css";
            $cssFileNames[] = "~/layouts/$defaultLayoutName/modules/GeoTools/resources/leaflet-vector-markers.css";
        }
        
        $cssInstances = $this->checkAndConvertCssStyles($cssFileNames);
        $headerCssInstances = array_merge($headerCssInstances, $cssInstances);
    
        return $headerCssInstances;
    }
    
    /*
     * Function to initialize the required data in smarty to display the List View Contents
    */
    public function initializeListViewContents(Vtiger_Request $request, Vtiger_Viewer $viewer) {
        global $log;
        $moduleName = $request->getModule();
        if($this->listViewName == 'noGeoCodes') {
            $cvId = 0;
            $this->settings['noGeoCodes'] = true;
        } else {
            $cvId = $this->listViewName;
        }
        $pageNumber = $request->get('page');
        $orderBy = $request->get('orderby');
        $sortOrder = $request->get('sortorder');
        if($sortOrder == "ASC"){
            $nextSortOrder = "DESC";
            $sortImage = "fa-sort-desc"; //To support open source sort icon
			$faSortImage = "fa-arrow-down";
        }else{
            $nextSortOrder = "ASC";
            $sortImage = "fa-sort-asc";
			$faSortImage = "fa-arrow-up";
        }
    
        if(empty ($pageNumber)){
            $pageNumber = '1';
        }

        // This this is where we need to override for GeoTools
        $listViewModel = GeoTools_ListView_Model::getInstance($moduleName, $cvId, $this->settings);
    
        $linkParams = array('LIST_MODULE'=>$moduleName, 'ACTION'=>$request->get('view'), 'CVID'=>$cvId, 'MODULE'=>$moduleName);

        $linkModels = $listViewModel->getListViewMassActions($linkParams);
    
        $pagingModel = new Vtiger_Paging_Model();
        $pagingModel->set('page', $pageNumber);
        if($this->listEntries > 0) {
            $pagingModel->set('limit', $this->listEntries);
        }
        $log->debug(print_r($pagingModel,true));
    
        if(!empty($orderBy)) {
            $listViewModel->set('orderby', $orderBy);
            $listViewModel->set('sortorder',$sortOrder);
        }
    
        $searchKey = $request->get('search_key');
        $searchValue = $request->get('search_value');
        $operator = $request->get('operator');
        if(!empty($operator)) {
            $listViewModel->set('operator', $operator);
            $viewer->assign('OPERATOR',$operator);
            $viewer->assign('ALPHABET_VALUE',$searchValue);
        }
        
        if(!empty($searchKey) && !empty($searchValue)) {
            $listViewModel->set('search_key', $searchKey);
            $listViewModel->set('search_value', $searchValue);
        }
        
        $searchParams = $request->get('search_params');
        if(empty($searchParams)) {
            $searchParams = array();
        }
        $transformedSearchParams = $this->transferListSearchParamsToFilterCondition($searchParams, $listViewModel->getModule());
        $listViewModel->set('search_params',$transformedSearchParams);


        //To make smarty to get the details easily accesible
        foreach($searchParams as $fieldListGroup){
            foreach($fieldListGroup as $fieldSearchInfo){
                $fieldSearchInfo['searchValue'] = $fieldSearchInfo[2];
                $fieldSearchInfo['fieldName'] = $fieldName = $fieldSearchInfo[0];
                $searchParams[$fieldName] = $fieldSearchInfo;
            }
        }
        
        // Create a dummy header for the dynamic distance column
        if(!$this->GeoToolsDistanceHeader){
            $this->GeoToolsDistanceHeader = new GeoTools_DistanceField_Model;
        }
        
        if(!$this->listViewHeaders){
            $this->listViewHeaders = $listViewModel->getListViewHeaders();
        }
        
        // Add our distance column to the Header array
        $this->listViewHeaders['distance'] = $this->GeoToolsDistanceHeader;
                
        if(!$this->listViewEntries){
            $this->listViewEntries = $listViewModel->getListViewEntries($pagingModel);
        }
        
        $noOfEntries = count($this->listViewEntries);
    
        $viewer->assign('LIST_MODULE', $moduleName);
    
        if(!$this->listViewLinks){
            $this->listViewLinks = $listViewModel->getListViewLinks($linkParams);
        }
        $viewer->assign('LISTVIEW_LINKS', $this->listViewLinks);
    
        $viewer->assign('LISTVIEW_MASSACTIONS', $linkModels['LISTVIEWMASSACTION']);
    
        $viewer->assign('PAGING_MODEL', $pagingModel);
        $viewer->assign('PAGE_NUMBER',$pageNumber);
    
        $viewer->assign('ORDER_BY',$orderBy);
        $viewer->assign('SORT_ORDER',$sortOrder);
        $viewer->assign('NEXT_SORT_ORDER',$nextSortOrder);
        $viewer->assign('SORT_IMAGE',$sortImage);
		$viewer->assign('FASORT_IMAGE',$faSortImage);
        $viewer->assign('COLUMN_NAME',$orderBy);
        $viewer->assign('SEARCH_DETAILS', $searchParams);
    
        $viewer->assign('LISTVIEW_ENTRIES_COUNT',$noOfEntries);
        $viewer->assign('LISTVIEW_HEADERS', $this->listViewHeaders);
        $viewer->assign('LISTVIEW_ENTRIES', $this->listViewEntries);

		$viewer->assign('VIEWID', $cvId);
    
        if (PerformancePrefs::getBoolean('LISTVIEW_COMPUTE_PAGE_COUNT', false)) {
            if(!$this->listViewCount){
                $this->listViewCount = $listViewModel->getListViewCount();
            }
            $totalCount = $this->listViewCount;
            $pageLimit = $pagingModel->getPageLimit();
            $pageCount = ceil((int) $totalCount / (int) $pageLimit);
    
            if($pageCount == 0){
                $pageCount = 1;
            }
            $viewer->assign('PAGE_COUNT', $pageCount);
            $viewer->assign('LISTVIEW_COUNT', $totalCount);
        }
            
        $viewer->assign('IS_MODULE_EDITABLE', $listViewModel->getModule()->isPermitted('EditView'));
        $viewer->assign('IS_MODULE_DELETABLE', $listViewModel->getModule()->isPermitted('Delete'));
    }
    
    
    /**
     * Function returns the number of records for the current filter
     * @param Vtiger_Request $request
     */
    function getRecordsCount(Vtiger_Request $request) {
        $moduleName = $request->get('geotoolsModule');
        $cvId = $request->get('viewname');
        $count = $this->getListViewCount($request);
    
        $result = array();
        $result['module'] = $moduleName;
        $result['viewname'] = $cvId;
        $result['count'] = $count;
    
        $response = new Vtiger_Response();
        $response->setEmitType(Vtiger_Response::$EMIT_JSON);
        $response->setResult($result);
        $response->emit();
    }
    
    /**
     * Function to get listView count
     * @param Vtiger_Request $request
     */
    function getListViewCount(Vtiger_Request $request){
        $moduleName = $request->get('geotoolsModule');
        $cvId = $request->get('viewname');
        if ($cvId == 'noGeoCodes'){
            $this->settings['noGeoCodes'] = true;
        }
        
        if(empty($cvId) || $cvId == 'noGeoCodes') {
            $cvId = '0';
        }
    
        $searchKey = $request->get('search_key');
        $searchValue = $request->get('search_value');
    
        $listViewModel = GeoTools_ListView_Model::getInstance($moduleName, $cvId, $this->settings);
        $searchParams = $request->get('search_params');
        if(empty($searchParams) && !is_array($searchParams)) {
            $searchParams = array();
        }
        
        $listViewModel->set('search_params',$this->transferListSearchParamsToFilterCondition($searchParams, $listViewModel->getModule()));
        
        $listViewModel->set('search_key', $searchKey);
        $listViewModel->set('search_value', $searchValue);
        $listViewModel->set('operator', $request->get('operator'));
    
        $count = $listViewModel->getListViewCount();
    
        return $count;
    }
    
    /**
     * Function to get the page count for list
     * @return total number of pages
     */
    function getPageCount(Vtiger_Request $request){
        $listViewCount = $this->getListViewCount($request);
        
        $pagingModel = new Vtiger_Paging_Model();
        if($this->listEntries > 0) {
            $pagingModel->set('limit', $this->listEntries);
        }
        $pageLimit = $pagingModel->getPageLimit();
        $pageCount = ceil((int) $listViewCount / (int) $pageLimit);
    
        if($pageCount == 0){
            $pageCount = 1;
        }
        $result = array();
        $result['page'] = $pageCount;
        $result['numberOfRecords'] = $listViewCount;

        $response = new Vtiger_Response();
        $response->setResult($result);
        $response->emit();
    }

	protected function assignCustomViews(Vtiger_Request $request, Vtiger_Viewer $viewer) {
		$allCustomViews = CustomView_Record_Model::getAllByGroup($request->getModule());
		if (!empty($allCustomViews)) {
			$viewer->assign('CUSTOM_VIEWS', $allCustomViews);
			$currentCVSelectedFields = array();
			foreach ($allCustomViews as $cat => $views) {
				foreach ($views as $viewModel) {
					if ($viewModel->getId() === $viewer->get_template_vars('VIEWID')) {
						$currentCVSelectedFields = $viewModel->getSelectedFields();
						$viewer->assign('CURRENT_CV_MODEL', $viewModel);
						break;
					}
				}
			}
		}
	}
    
    public function transferListSearchParamsToFilterCondition($listSearchParams, $moduleModel) {
        return Vtiger_Util_Helper::transferListSearchParamsToFilterCondition($listSearchParams, $moduleModel);
    }
}
