<?php
/*+***********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.1
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * Portions created by Libertus Solutions are Copyright (C) Libertus Solutions
 * All Rights Reserved.
 *************************************************************************************/

/**
 * GeoTools extended ListView Model Class
 */

class GeoTools_ListView_Model extends Vtiger_ListView_Model { 

    /**
     * Static Function to get the Instance of Vtiger ListView model for a given module and custom view
     * Overriden to get our custom QueryGenerator class
     * @param <String> $moduleName - Module Name
     * @param <Number> $viewId - Custom View Id
     * @param <Array> $settings - GeoTools parameters for Query Generator
     * @return Vtiger_ListView_Model instance
     */
    public static function getInstance($moduleName, $viewId='0', $settings = array()) {
        $db = PearDatabase::getInstance();
        $currentUser = vglobal('current_user');

        $modelClassName = Vtiger_Loader::getComponentClassName('Model', 'ListView', 'GeoTools'); // Force it to load our ListView Model
        $instance = new $modelClassName();
        $moduleModel = Vtiger_Module_Model::getInstance($moduleName);

        if (!class_exists('EnhancedQueryGenerator')) {
            $queryGenerator = new GeoTools_QueryGenerator_Model($moduleModel->get('name'), $currentUser, $settings);
        } else {
            $queryGenerator = new GeoTools_EnhancedQueryGenerator_Model($moduleModel->get('name'), $currentUser, $settings);
        }

        $customView = new CustomView();
        if (!empty($viewId) && $viewId != "0") {
            $queryGenerator->initForCustomViewById($viewId);

            //Used to set the viewid into the session which will be used to load the same filter when you refresh the page
            $viewId = $customView->getViewId($moduleName);
        } else {
            $viewId = $customView->getViewId($moduleName);
            if(!empty($viewId) && $viewId != 0) {
                $queryGenerator->initForDefaultCustomView();
            } else {
                $entityInstance = CRMEntity::getInstance($moduleName);
                $listFields = $entityInstance->list_fields_name;
                $listFields[] = 'id';
                $queryGenerator->setFields($listFields);
            }
        }
        $controller = new ListViewController($db, $currentUser, $queryGenerator);

        return $instance->set('module', $moduleModel)
                        ->set('query_generator', $queryGenerator)
                        ->set('listview_controller', $controller)
                        ->set('geoSettings', $settings);
    }

    /**
     * Static Function to get the Instance of Vtiger ListView model for a given module and custom view
     * @param <String> $value - Module Name
     * @param <Number> $viewId - Custom View Id
     * @return Vtiger_ListView_Model instance
     */
    public static function getInstanceForPopup($module) {
        $db = PearDatabase::getInstance();
        $currentUser = vglobal('current_user');

        $settings['popup'] = true;

        $modelClassName = Vtiger_Loader::getComponentClassName('Model', 'ListView', 'GeoTools');
        $instance = new $modelClassName();
        $moduleModel = Vtiger_Module_Model::getInstance($module);

        if (!class_exists('EnhancedQueryGenerator')) {
            $queryGenerator = new GeoTools_QueryGenerator_Model($moduleModel->get('name'), $currentUser, $settings);
        } else {
            $queryGenerator = new GeoTools_EnhancedQueryGenerator_Model($moduleModel->get('name'), $currentUser, $settings);
        }

        $listFields = $moduleModel->getPopupViewFieldsList();

        $listFields[] = 'id';
        $queryGenerator->setFields($listFields);

        $controller = new ListViewController($db, $currentUser, $queryGenerator);

        return $instance->set('module', $moduleModel)
                        ->set('query_generator', $queryGenerator)
                        ->set('listview_controller', $controller)
                        ->set('geoSettings', $settings);
    }

    /**
     * Function to get the list view entries - overridden to add distance and sortorder to the query.
     * @param Vtiger_Paging_Model $pagingModel
     * @return <Array> - Associative array of record id mapped to Vtiger_Record_Model instance.
     */
    public function getListViewEntries($pagingModel) {
        $db = PearDatabase::getInstance();

        $moduleName = $this->getModule()->get('name');
        $moduleFocus = CRMEntity::getInstance($moduleName);
        $moduleModel = Vtiger_Module_Model::getInstance($moduleName);

        $queryGenerator = $this->get('query_generator');
        $listViewContoller = $this->get('listview_controller');

        $geoSettings = $this->get('geoSettings');

        $geoSortOrder = $geoSettings['geotoolsSortOrder'];
        if($geoSortOrder == 'Nearest') {
            $geoSortOrder = 'ASC';
        } else {
            $geoSortOrder = 'DESC';
        }

        $searchParams = $this->get('search_params');
        if(empty($searchParams)) {
            $searchParams = array();
        }
        $glue = "";
        if(count($queryGenerator->getWhereFields()) > 0 && (count($searchParams)) > 0) {
            $glue = QueryGenerator::$AND;
        }
        $queryGenerator->parseAdvFilterList($searchParams, $glue);

        $searchKey = $this->get('search_key');
        $searchValue = $this->get('search_value');
        $operator = $this->get('operator');
        if(!empty($searchKey)) {
            $queryGenerator->addUserSearchConditions(array('search_field' => $searchKey, 'search_text' => $searchValue, 'operator' => $operator));
        }

        $orderBy = $this->getForSql('orderby');
        $sortOrder = $this->getForSql('sortorder');

        // No ORDER BY? GeoTools List view will be displayed by distance nearest first.
        // Unless it's for the Popup
        if(empty($orderBy) && empty($sortOrder) && !$geoSettings['popup'] && !$geoSettings['noGeoCodes']){
            $orderBy = 'distance';
            $sortOrder = $geoSortOrder;
        }

        // Add distance as a 2nd ORDER BY clause if not the first
        if($orderBy != 'distance' && !$geoSettings['popup'] && !$geoSettings['noGeoCodes']) {
            $geotoolsOrderby = ", distance " . $geoSortOrder;
        }

        if(!empty($orderBy)){
            $columnFieldMapping = $moduleModel->getColumnFieldMapping();
            $orderByFieldName = $columnFieldMapping[$orderBy];
            $orderByFieldModel = $moduleModel->getField($orderByFieldName);
            if($orderByFieldModel && $orderByFieldModel->getFieldDataType() == Vtiger_Field_Model::REFERENCE_TYPE){
                //IF it is reference add it in the where fields so that from clause will be having join of the table
                $queryGenerator = $this->get('query_generator');
                $queryGenerator->addWhereField($orderByFieldName);
                //$queryGenerator->whereFields[] = $orderByFieldName;
            }
        }
        $listQuery = $this->getQuery();

        $sourceModule = $this->get('src_module');
        if(!empty($sourceModule)) {
            if(method_exists($moduleModel, 'getQueryByModuleField')) {
                $overrideQuery = $moduleModel->getQueryByModuleField($sourceModule, $this->get('src_field'), $this->get('src_record'), $listQuery);
                if(!empty($overrideQuery)) {
                    $listQuery = $overrideQuery;
                }
            }
        }

        $startIndex = $pagingModel->getStartIndex();
        $pageLimit = $pagingModel->getPageLimit();

        if(!empty($orderBy)) {
            if($orderByFieldModel && $orderByFieldModel->isReferenceField()){
                $referenceModules = $orderByFieldModel->getReferenceList();
                $referenceNameFieldOrderBy = array();
                foreach($referenceModules as $referenceModuleName) {
                    $referenceModuleModel = Vtiger_Module_Model::getInstance($referenceModuleName);
                    $referenceNameFields = $referenceModuleModel->getNameFields();

                    $columnList = array();
                    foreach($referenceNameFields as $nameField) {
                        $fieldModel = $referenceModuleModel->getField($nameField);
                        $columnList[] = $fieldModel->get('table').$orderByFieldModel->getName().'.'.$fieldModel->get('column');
                    }
                    if(count($columnList) > 1) {
                        $referenceNameFieldOrderBy[] = getSqlForNameInDisplayFormat(array('first_name'=>$columnList[0],'last_name'=>$columnList[1]),'Users', '').' '.$sortOrder;
                    } else {
                        $referenceNameFieldOrderBy[] = implode('', $columnList).' '.$sortOrder ;
                    }
                }
                $listQuery .= ' ORDER BY '. implode(',',$referenceNameFieldOrderBy);
            } else if (!empty($orderBy) && $orderBy === 'smownerid') {
                $fieldModel = Vtiger_Field_Model::getInstance('assigned_user_id', $moduleModel); 
                if ($fieldModel->getFieldDataType() == 'owner') { 
                    $orderBy = 'COALESCE(CONCAT(vtiger_users.first_name,vtiger_users.last_name),vtiger_groups.groupname)'; 
                } 
                $listQuery .= ' ORDER BY '. $orderBy . ' ' .$sortOrder;
            } else {
                $listQuery .= ' ORDER BY '. $orderBy . ' ' .$sortOrder;
            }
        }

        // Add geoTools orderby if necessary
        if($geotoolsOrderby && !$geoSettings['popup']) {
            $listQuery .= ' ' . $geotoolsOrderby;
        }

        $viewid = ListViewSession::getCurrentView($moduleName);
        if(empty($viewid)) {
            $viewid = $pagingModel->get('viewid');
        }
        $_SESSION['lvs'][$moduleName][$viewid]['start'] = $pagingModel->get('page');

        ListViewSession::setSessionQuery($moduleName, $listQuery, $viewid);

        $listQuery .= " LIMIT $startIndex,".($pageLimit+1);

        $listResult = $db->pquery($listQuery, array());

        $listViewRecordModels = array();
        $listViewEntries =  $listViewContoller->getListViewRecords($moduleFocus,$moduleName, $listResult);

        $pagingModel->calculatePageRange($listViewEntries);

        if($db->num_rows($listResult) > $pageLimit){
            array_pop($listViewEntries);
            $pagingModel->set('nextPageExists', true);
        }else{
            $pagingModel->set('nextPageExists', false);
        }

        $index = 0;
        foreach($listViewEntries as $recordId => $record) {
            $rawData = $db->query_result_rowdata($listResult, $index++);
            $record['id'] = $recordId;
            $record['distance'] = $rawData['distance'];
            $rawData['hue'] = (hexdec(hash("adler32",$rawData['geopicklist'])) * 1.2) % 360;
            $listViewRecordModels[$recordId] = $moduleModel->getRecordFromArray($record, $rawData);
        }
        return $listViewRecordModels;
    }

    /**
     * Function to get the list view entries count
     * @return <Integer> - Count of records for this list view page.
     */
    public function getListViewCount() {

        $db = PearDatabase::getInstance();
        $queryGenerator = $this->get('query_generator');

        $searchParams = $this->get('search_params');
        if(empty($searchParams)) {
            $searchParams = array();
        }

        $glue = "";
        if(count($queryGenerator->getWhereFields()) > 0 && (count($searchParams)) > 0) {
            $glue = QueryGenerator::$AND;
        }
        $queryGenerator->parseAdvFilterList($searchParams, $glue);

        $searchKey = $this->get('search_key');
        $searchValue = $this->get('search_value');
        $operator = $this->get('operator');
        if(!empty($searchKey)) {
            $queryGenerator->addUserSearchConditions(array('search_field' => $searchKey, 'search_text' => $searchValue, 'operator' => $operator));
        }
        $moduleName = $this->getModule()->get('name');
        $moduleModel = Vtiger_Module_Model::getInstance($moduleName);

        $listQuery = $this->getQuery();

        $sourceModule = $this->get('src_module');
        if(!empty($sourceModule)) {
            $moduleModel = $this->getModule();
            if(method_exists($moduleModel, 'getQueryByModuleField')) {
                $overrideQuery = $moduleModel->getQueryByModuleField($sourceModule, $this->get('src_field'), $this->get('src_record'), $listQuery);
                if(!empty($overrideQuery)) {
                    $listQuery = $overrideQuery;
                }
            }
        }

        $listResult = $db->pquery($listQuery, array());

        return $db->num_rows($listResult); // Return a num_rows count instead. Not as quick but works!
    }

    /**
     * Function to get the list of Mass actions for the embedded List module
     * @param <Array> $linkParams
     * @return <Array> - Associative array of Link type to List of Vtiger_Link_Model instances for Mass Actions
     */
    public function getListViewMassActions($linkParams) {
        $currentUserModel = Users_Privileges_Model::getCurrentUserPrivilegesModel();
        $listViewModel = Vtiger_ListView_Model::getInstance($linkParams['LIST_MODULE']);
        return $listViewModel->getListViewMassActions($linkParams);
    }

}
