<?php
/*+***********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.1
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * Portions created by Libertus Solutions are Copyright (C) Libertus Solutions
 * All Rights Reserved.
 *************************************************************************************/

/**
 * GeoTools Module GeoCoder Class
 * Abstract - should be used as a parent of the provider class
 */
abstract class GeoTools_GeoCoder_Model {

    const USER_AGENT = 'GeoTools for vtiger CRM. Contact: enquiries@libertus.co.uk';
    const TIMEOUT = 10;

    public  $baseTable = 'libertus_geotools';
    public  $rate = 2500; // Default

    private $delay = 0;
    private $count = 0;
    private $httpAdapter = false; // In instantiation of GeoCoder set up best Http Adapter: Can use file_get_open or cURL
    private $configParams = array(); // Placeholder for specific Provider parameters such as APIKeys etc. Populated on object instantiation


    /**
     * function getCoords($locations)
     * This Orchestrates the GeoCoding process, passing formatted data
     * off to the relevant provider in the correct API.
     * $locations is an array of arrays of the address data and module record:
     * $location[]['module']
     * $location[]['crmid']
     * $location[]['field_street']
     * $location[]['field_city']
     * $location[]['field_state']
     * $location[]['field_postcode']
     * $location[]['field_country']
    **/
    public function getCoords($locations) {
        global $log;

        $retry = 0;

        // Iterate through the rows, geocoding each address
        $log->debug("Total No. of locations passed into " . __FUNCTION__ . ": " . sizeof($locations));
        foreach ($locations as $location) {
            $geocode_pending = true;

            while ($geocode_pending) {
                // If we have no interesting location data do not send for GeoCoding!
                if ($location['field_street'] == ""
                    && $location['field_postcode'] == ""
                    && $location['field_city'] == ""
                    && $location['field_state'] == "") {
                    $log->debug("No useful data for record " . $location['crmid'] . " Moving on");
                    break;
                }

                usleep($this->getDelay());

                // Call the geoCoder and get the result
                $result = $this->getGeocode($location);

                if(!$result) {
                    $log->debug("Unable to decode " . $location['crmid'] . "at this time.");
                    break;
                }

                $this->incrementCount();

                // All good!
                if ($result['status'] == 'OK') {
                    $geocode_pending = false; // No need to do this one again
                    if (!$this->save($result)) {
                        $log->debug("Unable to save " . print_r($result,true));
                        break;
                    }
                } else if ($result['status'] == 'UNABLE_TO_CONNECT') {
                    $log->debug("No network connectivity to Geocoder. Terminating Geocoder...");
                    exit;
                } else if ($result['status'] == 'TOO FAST') {

                    // sending geocodes too fast
                    //$log->info("Geocode too fast. Increasing delay");
                    $this->setDelay($this->getDelay() + 100000); 
                    $retry++;
                    // We've hit the daily limit
                    if ($retry > 3 ) return "LIMIT_REACHED";

                } else if ($result['status'] == "ZERO_RESULTS") {

                    $log->info("No results for " . print_r($location,true));
                    $log->info("Trying with reduced dataset");

                    $location['field_street'] = '';
                    $location['field_county'] = '';

                    $result = $this->getGeocode($location);

                    if($result) {
                        if ($result['status'] == 'OK') {
                            $geocode_pending = false; //skip to next location
                            $log->info("Successful geocode with reduced dataset");
                            if (!$this->save($result)) {
                                $log->debug("Unable to save " . print_r($result,true));
                                break;
                            }
                        } else if($result['status'] == "ZERO_RESULTS" || $result['status'] == "NO_LOCATION_DATA") {
                            $geocode_pending = false;
                            $log->info( "No result for this record. Saving with NULLs and moving on.");

                            // Save with null values so we don't keep trying to geocode again and again
                            $result['module'] = $location['module'];
                            $result['crmid'] = $location['crmid'];
                            $result['lat'] = null;
                            $result['lng'] = null;
                            if (!$this->save($result)) {
                                $log->debug("Unable to save empty record" . print_r($result,true));
                                break;
                            }
                            break;
                        }
                    } else {
                        $log->debug("No match for " . $location['crmid']);
                        //TODO We should probably flag these somehow...
                        //$log->debug("No match for $id => $address");
                    }
                } else if ($result['status'] == "NO_LOCATION_DATA") {
                    $geocode_pending = false; //skip to next location
                    $log->info("No location data for this record. Saving with NULLs and moving on.");

                    // Save with null values so we don't keep trying to geocode again and again
                    $result['module'] = $location['module'];
                    $result['crmid'] = $location['crmid'];
                    $result['lat'] = null;
                    $result['lng'] = null;
                    if (!$this->save($result)) {
                        $log->debug("Unable to save empty record" . print_r($result,true));
                        break;
                    }
                    break;
                } else {
                    $log->info("Failed to geocode " . $location['crmid'] . " Message was " . $result['status']);
                    $geocode_pending = false;
                }
            }

            flush();

            // Test if we've reached the rate limit
            // $log->debug("Rate: " . $this->getRate() . " Count: " . $this->getCount() );
            if($this->getCount() >= $this->getRate()) {
                $log->info("Rate Limit Reached of " . $this->getRate() . ". Terminating Geocoder...");
                exit;
            }
        }
        return "LIMIT_NOT_REACHED";
    }

    /**
     * function getGeocode()
     * Implementation specific function to call provider of Geocoding
     * services.
     *
     * $location is single record array
     *
     * returns location array with status, lat & lng values added
     *
    **/
    abstract public function getGeocode($location);

    /**
     * function save($location) saves the coords in the database.
     * $location array with has a return status and the lat & lng
     * added to it.
     *
    **/
    public function save($location, $locked=false) {
        global $log;

        if($location['crmid'] && $location['module'] && ($location['lat'] || $location['lat'] == null) && ($location['lng'] || $location['lng'] == null)) {

            $now = date("Y-m-d H:i:s");

            $query = "INSERT INTO libertus_geotools (module, lat, lng, locked, codedtime, crmid)
                        VALUES (?,?,?,?,?,?)
                      ON DUPLICATE KEY UPDATE
                        module = VALUES(module), lat = VALUES(lat), lng = VALUES(lng),
                        locked = VALUES(locked), codedtime = VALUES(codedtime), crmid = VALUES(crmid)";

            $db = PearDatabase::getInstance();
            $dbresult = $db->pquery($query, array($location['module'], $location['lat'], $location['lng'], $locked, $now, $location['crmid']));

            $log->debug("About to test db Connection");
            if(!$dbresult) {
                $log->debug("It was broken Let's try once again");
                $db->disconnect();
                unset($db);
                $db = new PearDatabase;
                $dbresult = $db->pquery($query, array($location['module'], $location['lat'], $location['lng'], $locked, $now, $location['crmid']));
                if(!$dbresult) {
                    // Let's move on anyway
                    return false;
                }
            } else {
                return true;
            }
        }
        // Not enough data to save a record
        return false;
    }

    public function isCoded($id) {
        $query = "SELECT COUNT(*) AS count
                    FROM $this->baseTable
                    WHERE crmid = ?
                    AND (lat IS NOT NULL
                         OR lng IS NOT NULL
                    )";

        $db = PearDatabase::getInstance();
        $result = $db->pquery($query, array($id));

        if($db->query_result($result, 0, 'count') == 1) {
            return true;
        } else {
            return false;
        }
    }

    public function isLocked($id) {
        $query = "SELECT locked
                  FROM $this->baseTable
                  WHERE crmid = ?";

        $db = PearDatabase::getInstance();
        $result = $db->pquery($query, array($id));

        if($db->query_result($result, 0, 'locked') == 1) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * function remove($id)
     * Removes a row from the geotools database by crmid
     * TODO This could be useful when triggered by an event handler
     * perhaps to remove geocodes when an entity record is deleted.
     *
    **/
    public function remove($id) {
        $query = "DELETE FROM $this->baseTable WHERE crmid = ?";

        $db = PearDatabase::getInstance();
        $result = $db->pquery($query, array($id));
        return $result;
    }

    protected function getDelay() {
        return $this->delay;
    }

    protected function setDelay($value) {
        $this->delay = intval($value);
    }

    public function setRate($rate) {
        if($rate) {
            $this->rate = intval($rate);
        }
    }

    public function getRate() {
        return $this->rate;
    }

    protected function incrementCount() {
        $this->count = $this->getCount() + 1;
    }

    protected function getCount() {
        return $this->count;
    }

    // Use cURL if available and we can't use file_get_contents
    protected function setHttpAdapter() {
        if($this->getHttpAdapter()) {
            return true;
        }
        if(ini_get('allow_url_fopen')) {
            $this->httpAdapter = 'fopen';
        } else if(function_exists('curl_version')) {
            $this->httpAdapter = 'curl';
        } else {
            throw new Exception('PHP is not compiled with CURL support and allow_url_fopen is disabled; giving up');
        }
    }

    /*
     * Function to set the params from the Provider class when object first instantiatied
     * Should only really be run once.
     */
    public function setParams($data) {
        if(!$this->getParams() && $data['config']) {
            $this->configParams = $data['config'];
        }
        // Set the important attributes if present in the Provider Config: rate and limit
        if($this->getParam('rate')) {
            $this->setRate($this->getParam('rate'));
        }
        if($this->getParam('limit')) {
            $limit = $this->getParam('limit') * 1000000; // Convert to Microseconds
            $this->setDelay($limit);
        }
    }

    public function getParams() {
        return $this->configParams;
    }

    public function getParam($name) {
        return $this->configParams[$name];
    }

    public function getHttpAdapter() {
        return $this->httpAdapter;
    }

    public function generateLocationsQuery($module, $fields, $type = 'New') {
        $db = PearDatabase::getInstance();

        $fieldids = array();

        $ent = CRMEntity::getInstance($module);
        $tabnameindex = $ent->tab_name_index;

        foreach($fields as $field) {
            if($field['fieldtype'] == 'field'){
                $fieldids[$field['fieldname']] = $field['fieldid'];
            }
        }

        $fieldstring = implode(",", $fieldids);
        $query = "SELECT tablename, columnname, fieldid
                  FROM vtiger_field
                  WHERE fieldid IN ($fieldstring)
                  ORDER BY field(fieldid, $fieldstring)";

        $result = $db->pquery($query, array());
        if($db->num_rows($result) > 1) {
            $i = 0;
            while($row = $db->fetch_array($result)) {
                $rows[] = $row['tablename'] . "." . $row['columnname'] . " AS " . array_search($row['fieldid'], $fieldids);
                $tables[] = $row['tablename'];
                $nodata[] = "(" . $row['tablename'] . "." . $row['columnname'] . " != '' AND " . $row['tablename'] . "." . $row['columnname'] . " IS NOT NULL)";
                $i++;
            }

            $tables = array_unique($tables);
            $len = count($tables);
            if($len > 1) {
                for($i = 1; $i < $len; $i++) {
                    $join .= " INNER JOIN " . $tables[$i] . " ON ";
                    $join .= $tables[$i] . "." . $tabnameindex[$tables[$i]] . " = ";
                    $join .= $tables[0] . "." . $tabnameindex[$tables[0]];
                }
            } else {
                $join = '';
            }

            $from = " FROM " . $tables[0];

            $join .= " LEFT JOIN $this->baseTable ON ";
            $join .= $tables[0] . "." . $tabnameindex[$tables[0]] . " = $this->baseTable.crmid";
            if($module != 'Users') {
                $join .= " INNER JOIN vtiger_crmentity ON ";
                $join .= $tables[0] . "." . $tabnameindex[$tables[0]] . " = vtiger_crmentity.crmid"; 
            }
            if(is_numeric($type)) {
                $where = " WHERE " . $tables[0] . "." . $tabnameindex[$tables[0]] . " = $type";
            } else if($type == 'New') {
                $where = " WHERE $this->baseTable.crmid IS NULL";
            } else if($type == 'Update' && $module != 'Users') {
                $where = " WHERE vtiger_crmentity.modifiedtime > $this->baseTable.codedtime AND $this->baseTable.locked = 0";
            } else if($type == 'Update' && $module == 'Users') {
                $where = " WHERE vtiger_users.date_modified > $this->baseTable.codedtime";
            } else {
                return false;
            }

            if($module != 'Users') {
               $where .= " AND vtiger_crmentity.deleted = 0";
            }

            $query = "SELECT " . $tables[0] . "." . $tabnameindex[$tables[0]] . " AS crmid, '" . $module ."' AS module, ";
            $query .= implode(', ', $rows);
            $query .= $from;
            $query .= $join;
            $query .= $where;

            // Possible assistance for systems where you must geocode in small batches
            // i.e. Shared Hosts that have very short cron/script timeouts
            if($this->getRate() < 500) {
                $query .= " AND ( " . implode(' OR ', $nodata) . ") ";
                $query .= " ORDER BY RAND() LIMIT " . $this->getRate() * 2;
            }

            return $query;
        }
        return false;
    }

    public function generateCompanyQuery() {
        $query = "SELECT organization_id AS crmid, 'CompanyDetails' AS module,
                  city AS field_city, state AS field_state,
                  code AS field_postcode, country AS field_country
                  FROM vtiger_organizationdetails
                  WHERE organization_id = 1";
        return $query;
    }

    public function getLocations($query) {
        $db = PearDatabase::getInstance();
        $result = $db->pquery($query, array());

        $count = $db->num_rows($result);
        if($count) {
            $locations = array();
            for($n = 0; $n < $count; $n++) {
                $locations[] = $db->raw_query_result_rowdata($result, $n); // Don't want htmlentities!
            }
            return $locations;
        }

        return false;
    }

    /**
     * Function to connect to geocoding Provider using fopen
    **/
    public function fopen_get_contents($url) {
        $options = array('http' => array('user_agent' => self::USER_AGENT,
                                         'timeout' => self::TIMEOUT));
        $context = stream_context_create($options);
        $data = file_get_contents($url, false, $context);
        return $data;
    }

    /*
     * Function to connect to geocoding Provider using cURL
    */
    public function curl_get_contents($url) {
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_HEADER, false);
        curl_setopt($curl, CURLOPT_USERAGENT, self::USER_AGENT);
        curl_setopt($curl, CURLOPT_TIMEOUT, self::TIMEOUT);
        $data = curl_exec($curl);
        curl_close($curl);
        return $data;
    }

    /*
     * function getIDsForDeletion()
     * Looks for crmids in the baseTable which do not have a corresponding
     * crmid in vtiger_crmentity
     * Returns an array of ids to remove
     */
    public function getIDsForDeletion() {
        $db = PearDatabase::getInstance();
        $query = "SELECT libertus_geotools.crmid
                  FROM libertus_geotools
                  LEFT JOIN vtiger_crmentity
                    ON vtiger_crmentity.crmid = libertus_geotools.crmid
                  WHERE libertus_geotools.module != 'Users'
                  AND vtiger_crmentity.crmid IS NULL";

        $result = $db->pquery($query, array());
        if($db->num_rows($result)){
            $ids = array();
            while($row = $db->fetch_array($result)) {
                $ids[] = $row['crmid'];
            }
            return $ids;
        }
        return '';
    }

    public static function getInstance($name = false){
        $moduleModel = new self();
        return $moduleModel;
    }
}
