<?php

/* +*******************************************************************************
 *  The contents of this file are subject to the vtiger CRM Public License Version 1.0
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * Portions created by Libertus Solutions are Copyright (C) Libertus Solutions.
 * All Rights Reserved.
 *
 * ******************************************************************************* */

require_once 'include/QueryGenerator/EnhancedQueryGenerator.php';

/**
 * Extended EnhancedQueryGenerator class for GeoTools
 */

class GeoTools_EnhancedQueryGenerator_Model extends EnhancedQueryGenerator {

    // GeoTools Params
    private $center_lat;
    private $center_lng;
    private $units;
    private $radius;
    private $constraint;
    private $popup;
    private $noGeocodes;

    public function __construct($module, $user, $settings) {
        $db = PearDatabase::getInstance();
        parent::__construct($module, $user);

        $this->center_lat = $this->getLatLng($settings['mapcentre'], 'lat');
        $this->center_lng = $this->getLatLng($settings['mapcentre'], 'lng');
        $this->units = $this->getUnits($settings['distance']);
        $this->radius = $settings['radius'];
        $this->constraint = $this->getConstraint($settings['constraint']);

        $this->popup = $settings['popup'];
        $this->noGeocodes = $settings['noGeoCodes'];

        $this->geoPicklist = $settings['icon_picklist'];
    }

    private function getLatLng($latlng, $type) {
        $latlng = explode(',', $latlng);
        if ($type == 'lat') {
            return $latlng[0];
        } else {
            return $latlng[1];
        }
    }

    private function getUnits($name) {
        if ($name == 'Kms') {
            return 6371;
        } else {
            return 3959;
        }
    }

    private function getConstraint($constraint) {
        if ($constraint == 'Inside') {
            return '<';
        } else {
            return '>';
        }
    }

    public function getCustomViewQueryById($viewId) {
        if ($viewId == 'noGeoCodes') {
            $this->initforNoGeocodes($viewId);
        } else {
            $this->initForCustomViewById($viewId);
        }

        return $this->getQuery();
    }

    public function getQuery() {
        if (empty($this->query)) {
            $conditionedReferenceFields = array();
            $allFields = array_merge($this->whereFields, $this->fields);
            foreach ($allFields as $fieldName) {
                if (in_array($fieldName, $this->referenceFieldList)) {
                    $moduleList = $this->referenceFieldInfoList[$fieldName];
                    foreach ($moduleList as $module) {
                        if (empty($this->moduleNameFields[$module])) {
                            $meta = $this->getMeta($module);
                        }
                    }
                } elseif (in_array($fieldName, $this->ownerFields)) {
                    $meta = $this->getMeta('Users');
                    $meta = $this->getMeta('Groups');
                }
            }

            $query = "SELECT ";
            $query .= $this->getSelectClauseColumnSQL();
            if (!$this->popup && !$this->noGeocodes) {
                $query .= $this->getGeoToolsDistance();
            }
            $query .= $this->getFromClause();
            if ($this->noGeocodes) {
                $query .= $this->getNoGeoCodesJoin();
            } else {
                $query .= $this->getGeoToolsJoin();
            }
            $query .= $this->getWhereClause();
            if ($this->noGeocodes) {
                $query .= $this->getNoGeoCodesWhere();
            }
            if (!$this->popup && !$this->noGeocodes) {
                $query .= $this->getGeoToolsHavingClause();
            }
            if($this->noGeocodes) {
                $query .= ")";
            }
            $this->query = $query;
            return $query;
        } else {
            return $this->query;
        }
    }

    public function getGeoToolsDistance() {

        $sql = ", libertus_geotools.crmid, libertus_geotools.module, libertus_geotools.lat, libertus_geotools.lng, 
                ROUND(
                        (" . $this->units . " 
                        * acos( 
                            cos( 
                                radians(" . $this->center_lat . ") 
                            ) * cos( 
                                radians( lat ) 
                            ) * cos( 
                                radians( lng ) - radians(" . $this->center_lng . ") 
                            ) + sin( 
                                radians(" . $this->center_lat . ")
                            ) * sin( 
                                radians( lat ) 
                            )
                        )
                    ), 2) 
                AS distance, libertus_geotools.locked ";

        if($this->geoPicklist) {
            $sql .= ", " . $this->getSQLColumn($this->geoPicklist) . " AS geopicklist ";
            if(!in_array($this->geoPicklist, $this->fields)) {
                $this->fields[] = $this->geoPicklist;
            }
        }

        return $sql;
    }

    public function getGeoToolsJoin() {
        if ($this->getModule() == 'Users') {
            $sql = " INNER JOIN libertus_geotools ON libertus_geotools.crmid = vtiger_users.id";
        } else {
            $sql = " INNER JOIN libertus_geotools ON libertus_geotools.crmid = vtiger_crmentity.crmid";
        }
        return $sql;
    }

    public function getNoGeoCodesJoin() {
        if ($this->getModule() == 'Users') {
            $sql = " LEFT JOIN libertus_geotools ON libertus_geotools.crmid = vtiger_users.id";
        } else {
            $sql = " LEFT JOIN libertus_geotools ON libertus_geotools.crmid = vtiger_crmentity.crmid";
        }
        return $sql;
    }

   public function getWhereClause() {
        $sql = parent::getWhereClause();
        $splitsql = '';
        if ($this->noGeocodes) {
            $splitquery = explode('AND', $sql, 2);
            $sql = $splitquery[0] . ' AND (' . $splitquery[1];
            //if $splitquery[1] is empty the query fails with multiple AND  
            /*
            if ($splitquery[1]) {
                $splitsql = " AND ";
            }*/
            $sql .= $splitsql;
        }else{
             $sql .= " AND (libertus_geotools.lat IS NOT NULL OR libertus_geotools.lng IS NOT NULL) ";
        }
        
        $this->whereClause = $sql;
        return $sql;
    }

    public function getNoGeoCodesWhere() {
        $sql = " AND (libertus_geotools.crmid IS NULL OR (libertus_geotools.lat IS NULL AND libertus_geotools.lng IS NULL)) ";
        return $sql;
    }
    

    public function getGeoToolsHavingClause() {
        $sql = " HAVING distance " . $this->constraint . " " . $this->radius . " ";
        return $sql;
    }

}

?>

