/*+***********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.1
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * Portions created by Libertus Solutions are Copyright (C) Libertus Solutions
 * All Rights Reserved.
 *************************************************************************************/
 
/*
 * GeoTools_List_Js: Main jQuery class file for user-intraction with the GeoTools page.
 * Overloading the normal Vtiger_List_Js class with modifications and 
 * new functions to handle the various GeoTools features.
 */
Vtiger_List_Js.extend('GeoTools_List_Js', {

    // Array to hold the map markers
    markers : [],
    
    // Base url
    urlParams : 'index.php?module=GeoTools&view=GeoTools',
    
    // Object to hold the map class based on the provider
    mapClass : false,
    
    // Object to hold the map object itself
    map : false,
    
    // Object to hold the keys for the Picklist derived icon colours
    key : {},
    
    // Called from the Geocoder_Js class on a map marker save after drag
    saveLocation : function(crmid, latlng, locked) {
        var thisInstance = this;
        var savingLocationMsg = app.vtranslate('JS_GEOTOOLS_SAVING_LOCATION');
        var geotoolsModule = jQuery('[name="geotoolsModule"]').val();
        var progressIndicatorElement = jQuery.progressIndicator({
            'message' : savingLocationMsg,
            'position' : 'html',
            'blockInfo' : {
                'enabled' : true
            }
        });
        
        var params = {
        'module' : 'GeoTools',
        'listModule' : geotoolsModule,
        'action' : 'SaveLocationAjax',
        'id' : crmid,
        'lat' : latlng.lat,
        'lng' : latlng.lng,
        'locked' : locked
        };
        
        AppConnector.request(params).then(
            function(data) {
                var msg = {};
                if(data['success']) {
                    msg.title = app.vtranslate('JS_GEOTOOLS_LOCATION_SAVED');
                    msg.text = app.vtranslate('JS_GEOTOOLS_LOCATION_UPDATED');
                    msg.type = 'success';
                    progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                } else {
                    msg.title = app.vtranslate('JS_GEOTOOLS_ERROR');;
                    msg.text = data['error']['message'];
                    msg.type = 'error';
                    progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                }
                Vtiger_Helper_Js.showMessage(msg);
            },
            function(error) {
                progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                //TODO : Handle error
            }
        );
    }

}, {

    init : function() {
        var className = 'GeoTools_' + jQuery('#mapProvider').val() + 'mapCode_Js';
        this.mapClass = window[className].getInstance();
        if($("select[name='geotoolsCenterOption']").val() == '4') {
                this.setLocation();
        }
    },
 
    // GeoTools: Overriden function to include GeoTools Search Params
    getDefaultParams : function() {
        var pageNumber = jQuery('#pageNumber').val();
        var module = 'GeoTools';
        var parent = app.getParentModuleName();
        var cvId = this.getCurrentCvId();
        var orderBy = jQuery('#orderBy').val();
        var sortOrder = jQuery("#sortOrder").val();
        var params = {
            'module': module,
            'parent' : parent,
            'page' : pageNumber,
            'view' : "List",
            'viewname' : cvId,
            'orderby' : orderBy,
            'sortorder' : sortOrder,
        };
        
        jQuery.extend(params, this.getGeoParams(false));

        var searchValue = this.getAlphabetSearchValue();

        if((typeof searchValue != "undefined") && (searchValue.length > 0)) {
            params['search_key'] = this.getAlphabetSearchField();
            params['search_value'] = searchValue;
            params['operator'] = "s";
        }
        params.search_params = JSON.stringify(this.getListSearchParams());
        return params;
    },
    
    getGeoParams : function(noCenter) {
    
        if(noCenter) {
            var geotoolsMapCenter = '';
        } else {
            var geotoolsMapCenter = jQuery('#geotoolsMapCenter').val();
        }
        
        var geotoolsModule = jQuery('[name="geotoolsModule"]').val();
        var geotoolsDistanceUnits = jQuery('[name="geotoolsDistanceUnits"]').val();
        var geotoolsRadius = jQuery('[name="geotoolsRadius"]').val();
        var geotoolsConstraint = jQuery('[name="geotoolsConstraint"]:checked').val();
        var geotoolsSortOrder = jQuery('[name="geotoolsSortOrder"]:checked').val();
        var geotoolsCenterOption = jQuery('[name="geotoolsCenterOption"]').val();
        var geoParams = {
            'geotoolsModule' : geotoolsModule,
            'geotoolsMapCenter' : geotoolsMapCenter,
            'geotoolsDistanceUnits' : geotoolsDistanceUnits,
            'geotoolsRadius' : geotoolsRadius,
            'geotoolsConstraint' : geotoolsConstraint,
            'geotoolsSortOrder' : geotoolsSortOrder,
            'geotoolsCenterOption' : geotoolsCenterOption
        };
        
        return geoParams;
    
    },
    
    /*
     * Use the navigator.geolocation function to get user's browser 
     * location and update the map centre before reload
     */
    setLocation : function(){
        var thisInstance = this;
        var progressIndicatorElement = jQuery.progressIndicator({
            'position' : 'html',
            'blockInfo' : {
                'enabled' : true
            }
        });
        if(navigator.geolocation){
        
            // timeout at 10000 milliseconds (10 seconds)
            var options = {timeout:10000};
            navigator.geolocation.getCurrentPosition(thisInstance.updateMapCenter.bind(this), 
                                                     thisInstance.errorHandler,
                                                     options);
            progressIndicatorElement.progressIndicator({'mode' : 'hide'});
        }else{
            progressIndicatorElement.progressIndicator({'mode' : 'hide'});
            var noNavigatorMessage = app.vtranslate('JS_GEOTOOLS_BROWSER_DOES_NOT_HAVE_NAVIGATOR');
            alert(noNavigatorMessage);
        }
    },
    
    // Update the #geotoolsMapCenter input with the browser's location 
    // then reload the page
    updateMapCenter : function(position) {
        var location = position.coords.latitude + ',' + position.coords.longitude;
        jQuery('#geotoolsMapCenter').val(location);
        urlParams = this.getDefaultParams();
        urlParams['geotoolsMapCenter'] = location;
        this.getListViewRecords(urlParams);
    },

    // If it doesn't work do this.
    errorHandler : function(err) {
        if(err.code == 1) {
            alert("Error: Access is denied!");
        } else if( err.code == 2) {
            alert("Error: Position is unavailable!");
        }
    },
    
    /*
     * Load the "Enter Address" popup and deal with the returned 
     * geocode entry (or not)
     *
     */
    enterAddress : function() {
        var thisInstance = this;
        var aDeferred = jQuery.Deferred();
        
        var progressIndicatorElement = jQuery.progressIndicator({
            'position' : 'html',
            'blockInfo' : {
                'enabled' : true
            }
        });
        
        var params = {};
        params['module'] = 'GeoTools';
        params['parent'] = app.getParentModuleName();
        params['view'] = 'AddressAjax';
        
        AppConnector.request(params).then(
            function(data) {
                var callBackFunction = function(data) {
                    var addressform = jQuery('#addressEntry');
                    
                    var params = app.validationEngineOptions;
                    params.onValidationComplete = function(addressform, valid){
                        if(valid) {
                            thisInstance.geocodeAddress(addressform);
                            return valid;
                        }
                    }
                    addressform.validationEngine(params);
                    
                    addressform.submit(function(e) {
                        e.preventDefault();
                    })
                }
                
                progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                app.showModalWindow(data,function(data){
                    if(typeof callBackFunction == 'function'){
                        callBackFunction(data);
                    }
                }, {'width':'400px'});
            },
            function(error) {
                progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                //TODO : Handle error
                aDeferred.reject(error);
            }
        );
        return aDeferred.promise();
    },
    
    /**
     * This function will geocode the address
     */
    geocodeAddress : function(form) {
        var thisInstance = this;
        var geocodingMessage = app.vtranslate('JS_GEOTOOLS_ATTEMPTING_TO_GEOCODE_ADDRESS');
        var progressIndicatorElement = jQuery.progressIndicator({
            'message' : geocodingMessage,
            'position' : 'html',
            'blockInfo' : {
                'enabled' : true
            }
        });
        
        var data = form.serializeFormData();
        data['module'] = 'GeoTools';
        data['parent'] = app.getParentModuleName();
        data['action'] = 'GeocodeAddressAjax';
        
        AppConnector.request(data).then(
            function(data) {
                if(data['success']) {
                    var location = data['result'][0];
                    $('#geotoolsMapCenter').val(location);
                    urlParams = thisInstance.getDefaultParams();
                    urlParams['geotoolsMapCenter'] = location;
                    progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                    app.hideModalWindow();
                    thisInstance.getListViewRecords(urlParams);
                } else {
                    alert(data['error']['message']);
                    progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                    thisInstance.enterAddress();
                }
            },
            function(error) {
                progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                //TODO : Handle error
            }
        );
    },
    
    getRecordsCount : function(){
        var aDeferred = jQuery.Deferred();
        var recordCountVal = jQuery("#recordsCount").val();
        if(recordCountVal != ''){
            aDeferred.resolve(recordCountVal);
        } else {
            var count = '';
            var cvId = this.getCurrentCvId();
            var module = app.getModuleName();
            var parent = app.getParentModuleName();
            var postData = {
                "module": 'GeoTools',
                "parent": parent,
                "view": "ListAjax",
                "viewname": cvId,
                "mode": "getRecordsCount"
            }
            
            jQuery.extend(postData, this.getGeoParams(false));

            var searchValue = this.getAlphabetSearchValue();
            if((typeof searchValue != "undefined") && (searchValue.length > 0)) {
                postData['search_key'] = this.getAlphabetSearchField();
                postData['search_value'] = this.getAlphabetSearchValue();
                postData['operator'] = "s";
            }

            postData.search_params = JSON.stringify(this.getListSearchParams());

            AppConnector.request(postData).then(
                function(data) {
                    var response = JSON.parse(data);
                    jQuery("#recordsCount").val(response['result']['count']);
                    count =  response['result']['count'];
                    aDeferred.resolve(count);
                },
                function(error,err){

                }
            );
        }

        return aDeferred.promise();
    },

    // When a non-geocded record is to be located via drag on the map. Paint a new marker and
    // let the user locate it.
    addRecordtoMap : function(crmid) {
        var tds = jQuery(".listViewEntries[data-id='"+crmid+"'] td");
        var headers = this.mapClass.getColumnHeaders();
        var latlng = this.mapClass.castLatlng(jQuery('#geotoolsMapCenter').val());
        var iconpath = $('#iconpath').val();
        var distance = '0';
        var locked = '';
                
        var col1value = tds[1].innerHTML;
        var col2value = tds[2].innerHTML;
        var col3value = tds[3].innerHTML;
        
        this.mapClass.createMarker(crmid, latlng, col1value, col2value, col3value, distance, iconpath, headers, locked);

    },

    /*
     * Function which will give you all the list view params
     * GeoTools customised to repaint the map after listview 
     * contents are updated
     */
    getListViewRecords : function(urlParams) {
        var aDeferred = jQuery.Deferred();
        if(typeof urlParams == 'undefined') {
            urlParams = {};
        }

        var thisInstance = this;
        
        var loadingMessage = jQuery('.listViewLoadingMsg').text();
        var progressIndicatorElement = jQuery.progressIndicator({
            'message' : loadingMessage,
            'position' : 'html',
            'blockInfo' : {
                'enabled' : true
            }
        });

        var defaultParams = thisInstance.getDefaultParams();
        var urlParams = jQuery.extend(defaultParams, urlParams);
        AppConnector.requestPjax(urlParams).then(
            function(data){
                progressIndicatorElement.progressIndicator({
                    'mode' : 'hide'
                })
                var listViewContentsContainer = jQuery('#listViewContents')
                listViewContentsContainer.html(data);
                app.showSelect2ElementView(listViewContentsContainer.find('select.select2'));
                app.changeSelectElementView(listViewContentsContainer);
                thisInstance.registerTimeListSearch(listViewContentsContainer);

                thisInstance.registerDateListSearch(listViewContentsContainer);
                thisInstance.calculatePages().then(function(data){
                    //thisInstance.triggerDisplayTypeEvent();
                    Vtiger_Helper_Js.showHorizontalTopScrollBar();

                    var selectedIds = thisInstance.readSelectedIds();
                    if(selectedIds != ''){
                        if(selectedIds == 'all'){
                            jQuery('.listViewEntriesCheckBox').each( function(index,element) {
                                jQuery(this).attr('checked', true).closest('tr').addClass('highlightBackgroundColor');
                            });
                            jQuery('#deSelectAllMsgDiv').show();
                            var excludedIds = thisInstance.readExcludedIds();
                            if(excludedIds != ''){
                                jQuery('#listViewEntriesMainCheckBox').attr('checked',false);
                                jQuery('.listViewEntriesCheckBox').each( function(index,element) {
                                    if(jQuery.inArray(jQuery(element).val(),excludedIds) != -1){
                                        jQuery(element).attr('checked', false).closest('tr').removeClass('highlightBackgroundColor');
                                    }
                                });
                            }
                        } else {
                            jQuery('.listViewEntriesCheckBox').each( function(index,element) {
                                if(jQuery.inArray(jQuery(element).val(),selectedIds) != -1){
                                    jQuery(this).attr('checked', true).closest('tr').addClass('highlightBackgroundColor');
                                }
                            });
                        }
                        thisInstance.checkSelectAll();
                    }
                    aDeferred.resolve(data);

                    // Let listeners know about page state change.
                    app.notifyPostAjaxReady();
                    
                    // GeoTools Customisation: Repaint the map
                    thisInstance.mapClass.showMap();
                    // Register the drag event handler
                    thisInstance.registerDragMarkerEvent();
                });
            },

            function(textStatus, errorThrown){
                aDeferred.reject(textStatus, errorThrown);
            }
        );
        return aDeferred.promise();
    },

    // Load the popup selector to choose new map centre
    openPopUp : function() {
        var thisInstance = this;
        var popupInstance = GeoTools_Popup_Js.getInstance();
        var params = {
            'module' : 'GeoTools',
            'src_module' : '',
            'src_field' : '',
            'src_record' : '' 
        };
        
        popupInstance.show(params,function(data) {
            var responseData = JSON.parse(data);
            thisInstance.getLatLngFromID(responseData);

        });
    },

    getLatLngFromID : function(responseData) {
        var thisInstance = this;
        var progressIndicatorElement = jQuery.progressIndicator({
            'position' : 'html',
            'blockInfo' : {
                'enabled' : true
            }
        });
        
        var data = responseData;
        data['module'] = 'GeoTools';
        data['parent'] = app.getParentModuleName();
        data['action'] = 'GetLatLngAjax';
        
        AppConnector.request(data).then(
            function(data) {
                if(data['success']) {
                    var location = data['result'][0];
                    jQuery('#geotoolsMapCenter').val(location);
                    urlParams = thisInstance.getDefaultParams();
                    urlParams['geotoolsMapCenter'] = location;
                    progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                    app.hideModalWindow();
                    thisInstance.getListViewRecords(urlParams);
                }
            },
            function(error) {
                progressIndicatorElement.progressIndicator({'mode' : 'hide'});
                //TODO : Handle error
            }
        );
    },
    
    /*
     * Function to register the GeoTools list view add record to map click event
     */
    registerAddRecordToMapClickEvent: function(){
        var thisInstance = this;
        var listViewContentDiv = this.getListViewContentContainer();
        listViewContentDiv.on('click','.addToMapButton',function(e){
            var elem = jQuery(e.currentTarget);
            var recordId = elem.closest('tr').data('id');
            thisInstance.addRecordtoMap(recordId);
            e.stopPropagation();
        });
    },

    /*
     * GeoTools: Change to the template layout means a slight change 
     * to the selector
     */
    registerEventForAlphabetSearch : function() {
        var thisInstance = this;
        //var listViewPageDiv = this.getListViewContentContainer();
        var alphabetSortingDiv = jQuery('.alphabetSorting');
        alphabetSortingDiv.on('click','.alphabetSearch',function(e) {
            var alphabet = jQuery(e.currentTarget).find('a').text();
            var cvId = thisInstance.getCurrentCvId();
            var AlphabetSearchKey = thisInstance.getAlphabetSearchField();
            var urlParams = {
                "viewname" : cvId,
                "search_key" : AlphabetSearchKey,
                "search_value" : alphabet,
                "operator" : 's',
                "page"  :   1
            }
            jQuery('#recordsCount').val('');
            //To Set the page number as first page
            jQuery('#pageNumber').val('1');
            jQuery('#pageToJump').val('1');
            jQuery('#totalPageCount').text("");
            thisInstance.getListViewRecords(urlParams).then(
                    function(data){
                        thisInstance.updatePagination();
                        //To unmark the all the selected ids
                        jQuery('#deSelectAllMsg').trigger('click');
                    },

                    function(textStatus, errorThrown){
                    }
            );
        });
    },

    /*
     * GeoTools: Trigger a reload of the ListView contents when GeoTools
     * search params change with whatever params are on-screen.
     */
    registerGeoToolsSearchEvents : function(){
        var thisInstance = this;
        var urlParams = GeoTools_List_Js.urlParams;
        
        jQuery("select[name='geotoolsDistanceUnits']")
            .add("input[name='geotoolsConstraint']")
            .add("input[name='geotoolsSortOrder']")
            .add("input[name='geotoolsRadius']").on('change', function() {
            thisInstance.getListViewRecords().then(
                function(data){
                    thisInstance.updatePagination();
                    //To unmark the all the selected ids
                    jQuery('#deSelectAllMsg').trigger('click');
                },
            function(textStatus, errorThrown){
                }
            )
        });
        // If the user changes module then we should clear all non-geo params back to defaults
        jQuery("select[name='geotoolsModule']").on('change', function() {
            var geoParams = $.param(thisInstance.getGeoParams(false));
            urlParams = urlParams+'&'+geoParams;
            document.location.assign(urlParams);
        });
        // Depending on the choice of map centre we need to clear the current lat/lng
        jQuery("select[name='geotoolsCenterOption']").on('change', function() {

            // Get Broswer's location and set the geotoolsMapCenter input
            if($(this).val() == '4') {
                thisInstance.setLocation();
            } else if($(this).val() == '5') {
                thisInstance.openPopUp();
            } else if($(this).val() == '6') {
                thisInstance.enterAddress();
            } else if($(this).val() == '100') {
                return;
            } else {
                geoParams = $.param(thisInstance.getGeoParams(true));
                urlParams = urlParams+'&'+geoParams;
                document.location.assign(urlParams);
            }
        });
    },
    
    /*
     * Funtion to insert the Non-Geocoded Custom Filter Option
     *
    */
    insertCustomView : function() {
        var option = '<option value="noGeoCodes" data-id="noGeoCodes">'+app.vtranslate('JS_LBL_GEOTOOLS_NON_GEOCODED_RECORDS')+'</option>';
        if(!$('select#customFilter option[value="noGeoCodes"]').length > 0) {
            $('select#customFilter').append(option);
        };
    },
    
    /*
     * Function to register the list view row click event
     * GeoTools: Re-written with an event to load the relevant map 
     * marker rather than navigate away to the target module
     */
    registerRowClickEvent: function(){
        var thisInstance = this;
        var listViewContentDiv = jQuery('.listViewContentDiv');
        // Unset existing click events first
        listViewContentDiv.off('click','.listViewEntries',function(e){
        });
        listViewContentDiv.on('click','.listViewEntries',function(e){
            if(jQuery(e.target, jQuery(e.currentTarget)).is('td:first-child')) return;
            if(jQuery(e.target).is('input[type="checkbox"]')) return;
            var elem = jQuery(e.currentTarget);         
            var markerNum = elem.index() -1;
            // Event marker is implmentation specific
            thisInstance.mapClass.addmarkerEvent(GeoTools_List_Js.markers[markerNum], 'click');
        });
    },
    
    /* So we can drag the center marker and trigger a reload
     *
     * This is not pretty. I Couldn't work out how to pass the event back
     * and forth successfully to the provider Javascript class.
     * TODO Replace the if(provider=='') test and have a common function in here
     * plus implementation specific code in the provider class.
    */
    registerDragMarkerEvent : function() {
        var thisInstance = this;
        var urlParams = '';
        
        // Center Marker must always be the last one added to the array
        var centerMarker = GeoTools_List_Js.markers[GeoTools_List_Js.markers.length - 1];
        var provider = $('#mapProvider').val();
        
        if(provider == 'Google') {
            google.maps.event.addListener(centerMarker, 'dragend', function (event) {
                var option = '<option value="100" selected="selected">'+app.vtranslate('JS_LBL_GEOTOOLS_DRAGGED_MARKER')+'</option>';
                var location = this.getPosition().lat() + ',' + this.getPosition().lng();
                $('#geotoolsMapCenter').val(location);
                if(!$('[name="geotoolsCenterOption"] option[value="100"]').length > 0) {
                    $('[name="geotoolsCenterOption"]').append(option);
                }
                urlParams = thisInstance.getDefaultParams();
                urlParams['geotoolsMapCenter'] = location;
                thisInstance.getListViewRecords(urlParams);
            });
        } else if (provider == 'OSM') {
            centerMarker.on('dragend', function (e) {
                var option = '<option value="100" selected="selected">'+app.vtranslate('JS_LBL_GEOTOOLS_DRAGGED_MARKER')+'</option>';
                var location = e.target.getLatLng();
                location = location['lat'] + ',' + location['lng'];
                $('#geotoolsMapCenter').val(location);
                    
                if(!$('[name="geotoolsCenterOption"] option[value="100"]').length > 0) {
                    $('[name="geotoolsCenterOption"]').append(option);
                }
                    
                urlParams = thisInstance.getDefaultParams();
                urlParams['geotoolsMapCenter'] = location;

                thisInstance.getListViewRecords(urlParams);
            });
        }
    },
    
    /*
     * GeoTools: Should load our own event handlers after the parent's
     * registerEvents() function has been called
     */
    registerEvents : function() {
        this._super();
        // GeoTools Event Registration
        this.registerGeoToolsSearchEvents();
        this.registerRowClickEvent();
        this.registerAddRecordToMapClickEvent();
        this.insertCustomView();
        // Load the map on first page load
        this.mapClass.showMap();
        // register this *after* the map has loaded
        this.registerDragMarkerEvent();
    }

});
